/*
 * ps2clusterattribs
 * Enums for PS2-specific cluster attribs 
 * 
 * (c) Criterion Software Limited
 */

/************************************************************************
 * File: PS2ClusterAttribs.h                                            *
 *                                                                      *
 * Purpose: Info used to communicate cluster format, etc during         *
 *          pipeline build time.                                        *
 ************************************************************************/

#ifndef _PS2CLUSTERATTRIBS_H
#define _PS2CLUSTERATTRIBS_H

#include "rwcore.h"


/* Only 32 bits are available, so use wisely */

/* RWPUBLIC */

/**
 * \ingroup rpworldp2sky2
 * Flags in PS2-specific cluster attributes
 * specifying data instancing behaviour. Clusters of interest to instancing
 * by MatInstance.csl or PS2Manager.csl will have at least one of the
 * following flags set 
 */
enum RxPS2ClusterAttrib
{
    CL_ATTRIB_REQUIRED    = 0x00000001, /**< Required. Data will be instanced and
                                         *   uploaded for this cluster */
    CL_ATTRIB_READ        = 0x00000002, /**< Read access required by user node(s)/callback(s) */
    CL_ATTRIB_WRITE       = 0x00000004, /**< Write access required by user node(s)/callback(s) */
    CL_ATTRIB_READWRITE   = 0x00000006, /**< Simply (CL_ATTRIB_READ  CL_ATTRIB_WRITE)
                                         * if either are set, the data is 'broken out'
                                         * (it's a contiguous array DMA'd by reference) */
    CL_ATTRIB_DONT_FILL   = 0x00000008, /**< Don't initialize the data. Will be
                                         * overwritten by user node(s)/callback(s) */
    CL_ATTRIB_PLACEHOLDER = 0x00000010, /**< VU mem place holder - space will be reserved on
                                         * VU1 but no data will be allocated CPU-side or uploaded.
                                         * NOTE: REQUIRED and PLACEHOLDER should not both be set.
                                         * If both are, REQUIRED overrides PLACEHOLDER */

    /* RWPUBLICEND
     * This makes things easier for us. Extra logic will permit an ugly hack to have opaque.
     * If neither read or write are set, CL_ATTRIB_OPAQUE is automatically set
     * RWPUBLIC */
    CL_ATTRIB_OPAQUE      = 0x00002000, /**<For internal use */
    /* RWPUBLICEND
     * CL_ATTRIB_STATIC - One day we will split short- and long-lived
     * data to improve memory efficiency in multibuffered cases
     * RWPUBLIC */
    CL_ATTRIB_STATIC      = 0x00008000, /**<For internal use */

    rxRXPS2CLUSTERATTRIBSFORCEENUMSIZEINT = RWFORCEENUMSIZEINT
};
typedef enum RxPS2ClusterAttrib RxPS2ClusterAttrib;

/**
 * \ingroup rpworldp2sky2
 *  Flags in PS2-specific cluster attributes
 * defining the format of cluster data in DMA packets. These formats get
 * expanded by the VIF on upload to VU1. 
 */
enum RxPS2ClusterFormatAttrib
{
    CL_S32   = 0x60000000, /**< See the EE user manual section 4.3.4 */
    CL_V2_32 = 0x64000000, /**< See the EE user manual section 4.3.4 */
    CL_V2_16 = 0x65000000, /**< See the EE user manual section 4.3.4 */
    CL_V3_32 = 0x68000000, /**< See the EE user manual section 4.3.4 */
    CL_V4_32 = 0x6c000000, /**< See the EE user manual section 4.3.4 */
    CL_V4_16 = 0x6d000000, /**< See the EE user manual section 4.3.4 */
    CL_V4_8  = 0x6e000000, /**< See the EE user manual section 4.3.4 */

    CL_USN   = 0x00004000, /**<Treat the data as unsigned */

    CL_TYPE_MASK = 0xff004000, /**<We allocate 9 bits for use as a format specifier,
                                * matching the bits in the VIF UNPACK command */
    rxRXPS2CLUSTERFORMATATTRIBSFORCEENUMSIZEINT = RWFORCEENUMSIZEINT
};
typedef enum RxPS2ClusterFormatAttrib RxPS2ClusterFormatAttrib;


/**
 * \ingroup rpworldp2sky2
 *  Values specifying the type of
 * PS2-specific clusters. See \ref RxPipelineNodePS2MatInstanceGenerateCluster
 * and \ref RxPipelineNodePS2ManagerGenerateCluster. There are only eight
 * possible clusters uploadable to VU1 on PS2 currently. The first four
 * are always uploaded to VU1 in the order listed VU (some can be missing)
 * and the user data clusters come after that. The data format of the four
 * standard clusters is fixed, changing the attributes value for
 * that won't work 
 */
enum RxPS2ClusterType
{
    CL_XYZ     = 0, /**< Vertex positions, fixed to CL_V3_32 format */
    CL_UV      = 1, /**< Vertex texture coordinates, fixed to CL_V2_32 format */
    CL_UV2     = 2, /**< Vertex texture coordinates, fixed to CL_V4_32 format */
    CL_RGBA    = 3, /**< Vertex prelight colors, fixed to (CL_V4_8  CL_USN) format */
    CL_NORMAL  = 4, /**< Vertex normals, fixed to CL_V4_8 format */
    CL_USER1   = 5, /**< First user data cluster, modifiable attributes */
    CL_USER2   = 6, /**< Second user data cluster, modifiable attributes */
    CL_USER3   = 7, /**< Third user data cluster, modifiable attributes */
    CL_USER4   = 8, /**< Fourth user data cluster, modifiable attributes */
    CL_MAXCL   = 9, /**< Nine clusters supported at the moment */
/* RWPUBLICEND */
    /* NOTE: if during morphing (with FASTMORPH defined), XYZ is present,
     * then a second XYZ (for the end morphTarget) will be placed in slot
     * CL_MAXCL. Similarly, if NORMALs are present, a second set will be
     * placed in (CL_MAXCL+1). */
/* RWPUBLIC */
    rxPS2CLUSTERTYPFORCEENUMSIZEINT = RWFORCEENUMSIZEINT
};
typedef enum RxPS2ClusterType RxPS2ClusterType;

/* RWPUBLICEND */

#endif /* !_PS2CLUSTERATTRIBS_H */

