
/***************************************************************************
 *                                                                      *
 * Module  : bamateri.h                                                    *
 *                                                                         *
 * Purpose : Device layer                                                  *
 *                                                                         *
 **************************************************************************/

#ifndef RWMATERI_H
#define RWMATERI_H

/****************************************************************************
 Includes
 */

#include <rwcore.h>

/****************************************************************************
 Defines
 */

/* RWPUBLIC */

/*
 * Handling surface materials
 * Materials describe how things are to appear when rendered
 *
 * Copyright (c) 1998 Criterion Software Ltd.
 */

/****************************************************************************
 Global Types
 */

/* RWPUBLICEND */
typedef struct rpMaterialGlobals rpMaterialGlobals;
struct rpMaterialGlobals
{
    RwFreeList           *matFreeList;
};

/* RWPUBLIC */

typedef struct RpMaterialChunkInfo RpMaterialChunkInfo;
typedef struct RpMaterialChunkInfo _rpMaterial;

struct RpMaterialChunkInfo
{
    RwInt32             flags;  /**<  Material flags - unused currently - 
                                   for future expansion */
    RwRGBA              color;  /**<  Colour of material. */
    RwInt32             unused;  /**<  Not used */
    RwBool              textured;  /**<  Are we textured? */
#ifdef RXPIPELINE
    RwSurfaceProperties surfaceProps;   /**<  Surface properties - only relevant for PowerPipe */
                                        /**<  Included in PipeLine 1 for file compatibility */
#else /* RXPIPELINE */
    RwSurfaceProperties ignoredSurfaceProps;
#endif /* RXPIPELINE */
};

#if (!defined(RwMaterialAssign))
#define RwMaterialAssign(_target, _source)             \
    ( *(_target) = *(_source) )
#endif /* (!defined(RwMaterialAssign)) */

/**
 * \ingroup rpworlddatatypes
 * \typedef RpMaterial 
 * Material object. This should be 
 * considered an opaque type. Use the RpMaterial API functions to access.
 */
typedef struct RpMaterial RpMaterial;

#if (!defined(DOXYGEN))
struct RpMaterial
{
        RwTexture           *texture; /**< texture */
        RwRGBA              color; /**< color */              
#if (defined(RXPIPELINE))
        RxPipeline          *pipeline; /**< pipeline */     

        RwSurfaceProperties surfaceProps; /**< surfaceProps */
#else /* (defined(RXPIPELINE)) */
        /* Not used by the Pipeline1 rendering
         * but present so pp files can be loaded and
         * saved and the values not destroy
         */
        RwSurfaceProperties ignoredSurfaceProps;    

#endif /* (defined(RXPIPELINE)) */

        RwInt16             refCount;          /* C.f. rwsdk/world/bageomet.h:RpGeometry */
        RwInt16             pad;
};
#endif /* (!defined(DOXYGEN)) */

/**
 * \ingroup rpworlddatatypes
 * \typedef RpMaterialCallBack
 \ref RpMaterialCallBack 
 * represents the function called from \ref RpGeometryForAllMaterials and 
 * \ref RpWorldForAllMaterials for all materials referenced by polygons in a 
 * given geometry. This function should return a pointer to the current 
 * material to indicate success. The callback may return NULL to terminate 
 * further callbacks on the materials.
 * 
 * \param  material   Pointer to the current material
 * \param  data  Pointer to developer-defined data structure.
 * 
 * \return Pointer to the current material.
 */
typedef RpMaterial *(*RpMaterialCallBack)(RpMaterial *material, void *data);

/****************************************************************************
 Function prototypes
 */

#ifdef    __cplusplus
extern "C"
{
#endif                          /* __cplusplus */

/* RWPUBLICEND */

/* Opening and closing materials */
extern void *_rpMaterialClose(void *instance, RwInt32 offset, RwInt32 size);
extern void *_rpMaterialOpen(void *instance, RwInt32 offset, RwInt32 size);

/* Setting default surface properties - LEGACY-SUPPORT hack for older files */
extern void _rpMaterialSetDefaultSurfaceProperties( const RwSurfaceProperties *surfaceProps );

/* RWPUBLIC */

/* Creating, destroying and referencing materials */
extern RpMaterial *RpMaterialCreate(void);
extern RwBool RpMaterialDestroy(RpMaterial *material);
extern RpMaterial *RpMaterialAddRef(RpMaterial *material);
extern RpMaterial *RpMaterialClone(RpMaterial *material);

/* Textures */
extern RpMaterial *RpMaterialSetTexture(RpMaterial *material, RwTexture *texture);
extern RwTexture *RpMaterialGetTexture(const RpMaterial *material);

/* Setting and getting colors */
extern RpMaterial *RpMaterialSetColor(RpMaterial *material, const RwRGBA *color);
extern const RwRGBA *RpMaterialGetColor(const RpMaterial *material);

/* Setting and getting surface properties */
extern RpMaterial *RpMaterialSetSurfaceProperties(RpMaterial * material, const RwSurfaceProperties * surfaceProperties);
extern const RwSurfaceProperties *RpMaterialGetSurfaceProperties(const RpMaterial * material);

/* Attaching toolkits */
extern RwInt32 RpMaterialRegisterPlugin(RwInt32 size, RwUInt32 pluginID,
                                        RwPluginObjectConstructor constructCB,
                                        RwPluginObjectDestructor destructCB,
                                        RwPluginObjectCopy copyCB);
extern RwInt32 RpMaterialRegisterPluginStream(RwUInt32 pluginID,
                                              RwPluginDataChunkReadCallBack readCB,
                                              RwPluginDataChunkWriteCallBack writeCB,
                                              RwPluginDataChunkGetSizeCallBack getSizeCB);
extern RwInt32 RpMaterialSetStreamAlwaysCallBack(
                        RwUInt32 pluginID,
                        RwPluginDataChunkAlwaysCallBack alwaysCB);
extern RwInt32 RpMaterialGetPluginOffset(RwUInt32 pluginID);
extern RwBool RpMaterialValidatePlugins(const RpMaterial *material);

/* Binary format */
extern RwUInt32 RpMaterialStreamGetSize(const RpMaterial *material);
extern RpMaterial *RpMaterialStreamRead(RwStream *stream);
extern const RpMaterial *RpMaterialStreamWrite(const RpMaterial *material, RwStream *stream);
extern RpMaterialChunkInfo *
RpMaterialChunkInfoRead(RwStream *stream, RpMaterialChunkInfo *materialChunkInfo, RwInt32 *bytesRead);

#ifdef    __cplusplus
}
#endif                          /* __cplusplus */

/* RWPUBLICEND */

#endif /* RWMATERI_H */

