
/***************************************************************************
 *                                                                         *
 * Module  : basync.c                                                      *
 *                                                                         *
 * Purpose : Syncing up the world                                          *
 *                                                                         *
 **************************************************************************/

/****************************************************************************
 Includes
 */

#include <stdlib.h>

#include "batypes.h"
#include "balibtyp.h"
#include "badebug.h"
#include "batypehf.h"

#include "bamatrix.h"
#include "baframe.h"
#include "bavector.h"

/* */

#include "basync.h"

#if (!defined(DOXYGEN))
static const char  __RWUNUSED__ rcsid[] =
   "@@(#)$Id: basync.c,v 1.59 2001/07/16 15:40:46 johns Exp $";
#endif /* (!defined(DOXYGEN)) */

#if defined (__MWERKS__)
#if (defined(RWVERBOSE))
#pragma message (__FILE__ "/" _SKY_EXPAND(__LINE__) ": __MWERKS__ == " _SKY_EXPAND(__MWERKS__))
#endif /* (defined (__MWERKS__)) */
#if (__option (global_optimizer))
#pragma always_inline off
#endif /* (__option (global_optimizer)) */
#endif /*  defined (__MWERKS__) */

/****************************************************************************
 Local Types
 */

/****************************************************************************
 Local (Static) Prototypes
 */

/****************************************************************************
 Local Defines
 */

/****************************************************************************
 Globals (across program)
 */

/****************************************************************************
 Local (static) Globals
 */

/* !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

   Debugging

   !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! */

/* !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

   Opening/Closing

   !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! */

/****************************************************************************
 _rwSyncObject

 On entry   : Object to sync, user data (not used)
 On exit    : Object pointer
 */

static RwObject    *
SyncObject(RwObject * object, void * __RWUNUSED__ data)
{
    RWFUNCTION(RWSTRING("SyncObject"));
    RWASSERT(object);

    rwObjectHasFrameSync((RwObjectHasFrame *) object);

    RWRETURN(object);
}

/****************************************************************************
 _rwFrameSyncHierarchyRecurse

 Syncs a frames hierarchy and objects attached (recursively)

 On entry   : Root frame of the hierarchy to sync, dirty
 On exit    :
 */

static void
FrameSyncHierarchyRecurse(RwFrame *frame, RwInt32 dirty)
{
    RWFUNCTION(RWSTRING("FrameSyncHierarchyRecurse"));

    /* NULL is a valid; termination condition */
    if (frame)
    {
        RwInt32 childDirty = dirty |
            rwObjectTestPrivateFlags(frame, (rwFRAMEPRIVATESUBTREESYNCLTM|
                                             rwFRAMEPRIVATESUBTREESYNCOBJ));

        RWASSERTISTYPE(frame, rwFRAME);

        if (childDirty)
        {
            if (childDirty & rwFRAMEPRIVATESUBTREESYNCLTM)
            {
                /* Work out the new local transformation matrix */
                RwMatrixMultiply(&frame->ltm,
                                 &frame->modelling,
                                 &((_rwFrameGetParent(frame))->ltm));
            }

            /* We're about to sync the objects concerned, so we can reset the dirty bit and sync bit */
            rwObjectSetPrivateFlags(frame, 
                                    rwObjectGetPrivateFlags(frame) &
                                    ~(rwFRAMEPRIVATESUBTREESYNCLTM |
                                      rwFRAMEPRIVATEHIERARCHYSYNCLTM |
                                      rwFRAMEPRIVATEHIERARCHYSYNCOBJ |
                                      rwFRAMEPRIVATESUBTREESYNCOBJ));

            /* If we got here, we always need to sync the objects */
            RwFrameForAllObjects(frame, SyncObject, NULL);
        }

        /* Depth first...
         * Child has dirty status including this frame,
         * sibling has dirty status of parent (parameter in)
         */
        FrameSyncHierarchyRecurse(frame->child, childDirty);
        FrameSyncHierarchyRecurse(frame->next, dirty);
    }

    RWRETURNVOID();
}

/****************************************************************************
 _rwFrameSyncHierachy

 Syncs a frames hierarchy and objects attached

 On entry   : Root frame of the hierarchy to sync
 On exit    :
 */

static void
FrameSyncHierarchy(RwFrame *frame)
{
    RwInt32             dirty;

    RWFUNCTION(RWSTRING("FrameSyncHierarchy"));
    RWASSERT(frame);
    RWASSERTISTYPE(frame, rwFRAME);

    /* Is a sync required */
    dirty = rwObjectTestPrivateFlags(frame, (rwFRAMEPRIVATESUBTREESYNCLTM |
                                             rwFRAMEPRIVATESUBTREESYNCOBJ));

    /* Then clear the dirty bits */
    rwObjectSetPrivateFlags(frame, 
                            rwObjectGetPrivateFlags(frame) &
                            ~(rwFRAMEPRIVATESUBTREESYNCLTM |
                              rwFRAMEPRIVATEHIERARCHYSYNCLTM |
                              rwFRAMEPRIVATEHIERARCHYSYNCOBJ |
                              rwFRAMEPRIVATESUBTREESYNCOBJ));

    if (dirty)
    {
        if (dirty & rwFRAMEPRIVATESUBTREESYNCLTM)
        {
            /* Root of hierarchy has no parent matrix - different from rest of hierarchy */
            RwMatrixCopy(&frame->ltm, &frame->modelling);
        }

        /* Handle objects in the world (only if matrix is dirty) */
        RwFrameForAllObjects(frame, SyncObject, NULL);
    }

    /* Do the children */
    FrameSyncHierarchyRecurse(frame->child, dirty);

    /* All done */
    RWRETURNVOID();
}

/****************************************************************************
 FrameSyncHierarchyLTMRecurse

 Syncs a frames local transformation matrix hierarchy but not objects
 positions (recursively)

 On entry   : Root frame of the hierarchy to sync, dirty
 On exit    :
 */

static void
meSyncHierarchyLTMRecurse(RwFrame * frame, RwInt32 dirty)
{
    RWFUNCTION(RWSTRING("meSyncHierarchyLTMRecurse"));

    /* NULL is a valid; termination condition */
    if (frame)
    {
        RwInt32 childDirty = dirty |
            rwObjectTestPrivateFlags(frame, rwFRAMEPRIVATESUBTREESYNCLTM);

        RWASSERTISTYPE(frame, rwFRAME);

        if (childDirty)
        {
            /* Work out the new local transformation matrix */
            RwMatrixMultiply(&frame->ltm, 
                             &frame->modelling,
                             &((_rwFrameGetParent(frame))->ltm));

            /* We've updated the LTM, so say so */
            rwObjectSetPrivateFlags(frame, rwObjectGetPrivateFlags(frame) &
                                    (~rwFRAMEPRIVATESUBTREESYNCLTM));
        }

        /* Depth first */
        /* Child has dirty status including this frame,
         * sibling has dirty status of parent (parameter in)
         */
        meSyncHierarchyLTMRecurse(frame->child, childDirty);
        meSyncHierarchyLTMRecurse(frame->next, dirty);
    }

    RWRETURNVOID();
}

/****************************************************************************
 _rwFrameSyncHierachyLTM

 Syncs a frames hierarchy - but does not sync the objects attached to
 the frame (this will be performed later with the FrameSyncHierarchy
 function, presumably when the begin camera update is performed).

 On entry   : Root frame
 On exit    :
 */

void
_rwFrameSyncHierarchyLTM(RwFrame * frame)
{
    RwInt32             dirty;

    RWFUNCTION(RWSTRING("_rwFrameSyncHierarchyLTM"));
    RWASSERT(frame);
    RWASSERTISTYPE(frame, rwFRAME);

    /* Is a sync required */
    dirty = rwObjectTestPrivateFlags(frame, rwFRAMEPRIVATESUBTREESYNCLTM);

    /* We'll update the LTMs, so say so */
    rwObjectSetPrivateFlags(frame, rwObjectGetPrivateFlags(frame) &
                            ~(rwFRAMEPRIVATESUBTREESYNCLTM |
                              rwFRAMEPRIVATEHIERARCHYSYNCLTM));

    if (dirty)
    {
        /* Root of hierarchy has no parent matrix - different from rest of hierarchy */
        RwMatrixCopy(&frame->ltm, &frame->modelling);
    }

    /* Do the children */
    meSyncHierarchyLTMRecurse(frame->child, dirty);

    /* All done */
    RWRETURNVOID();
}

/****************************************************************************
 _rwFrameSyncDirty

 Syncs all dirty frame hierarchies in the system.

 On entry   : None
 On exit    : TRUE on success
 */

RwBool
_rwFrameSyncDirty(void)
{
    RwLLLink           *lpFrameCur, *lpFrameEnd;

    RWFUNCTION(RWSTRING("_rwFrameSyncDirty"));

    lpFrameCur = rwLinkListGetFirstLLLink(&RWSRCGLOBAL(dirtyFrameList));
    lpFrameEnd = rwLinkListGetTerminator(&RWSRCGLOBAL(dirtyFrameList));

    while (lpFrameCur != lpFrameEnd)
    {
        RwFrame *rootFrame =
            rwLLLinkGetData(lpFrameCur, RwFrame, inDirtyListLink);

        /* Sync up the hierarchy */
        FrameSyncHierarchy(rootFrame);

        /* Onto the next */
        lpFrameCur = rwLLLinkGetNext(lpFrameCur);
    }

    /* We have dealt with them all -> everybody's happy ! */
    rwLinkListInitialize(&RWSRCGLOBAL(dirtyFrameList));

    /* All done */
    RWRETURN(TRUE);
}

#if defined (__MWERKS__)
#if (defined(RWVERBOSE))
#pragma message (__FILE__ "/" _SKY_EXPAND(__LINE__) ": __MWERKS__ == " _SKY_EXPAND(__MWERKS__))
#endif /* (defined (__MWERKS__)) */
#if (__option (global_optimizer))
#pragma always_inline on
#endif /* (__option (global_optimizer)) */
#endif /*  defined (__MWERKS__) */
