/*----------------------------------------------------------------------*
 *                                                                      *
 * Module  :                                                            *
 *                                                                      *
 * Purpose :                                                            *
 *                                                                      *
 * FX      :                                                            *
 *----------------------------------------------------------------------*/

/**
 * \defgroup rpmatfxgcn RpMatFx on GameCube
 * \ingroup rpmatfx
 * 
 * Material Effects on GameCube
 */

/** 
 * \ingroup rpmatfxgcn
 *
 * \page rpmatfxgcnoverview Overview of RpMatFX on GameCube
 * 
 * RpMatFX on GameCube is provided for PS2 compatibility. It is in no way a GameCube
 * optimized pipeline. 
 * No rework of art resources is needed to run RpMatFX on the GameCube.
 *
 * \par Environment Mapping
 *
 * Environment mapping is done in one pass using using Hardware generated 
 * texture coordinates. 
 * \par Known bugs
 * The shininess factor is not implemented on the GameCube emulator.
 * Due to a limitation in the current GameCube hardware, a base texture
 * must be defined in order to have the Environment Mapping working
 *
 * \par Bump Mapping
 *
 * Bump Mapping is currently not implemented on GameCube. This will be fixed
 * shortly in a forthcoming release.
 *
 * \par Bump Environment Mapping
 *
 * Bump Environment Mapping is currently not implemented on GameCube. This will
 * be fixed shortly in a forthcoming release.
 *
 * \par Dual Pass
 *
 * Dual Pass on Gamecube uses a one pass approach. 
 * \par Known bug
 * Due to a limitation in the current GameCube hardware, the two textures
 * must be defined in order to have Dual Pass working
 *
 * \see \ref fixedfunctionpipeline
 *
 */


/*----------------------------------------------------------------------*
 *-   Includes                                                         -*
 *----------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/*==== GC libs includes =====*/
#include"dolphin.h"

/*==== RW libs includes =====*/
#include "rpplugin.h"
#include <rpdbgerr.h>
#include <rwcore.h>
#include <rpmatfx.h>

#include "effectpipes.h"


/*----------------------------------------------------------------------*
 *-   Local Types                                                      -*
 *----------------------------------------------------------------------*/
/* These types shouldn't be here, but it's still better than having to
   include p2 headers */

typedef struct __RpGameCubeFilterParams _RpGameCubeFilterParams;
struct __RpGameCubeFilterParams
{
    GXTexFilter min;
    GXTexFilter max;
};

typedef struct _rwDlStateCache _rwDlStateCache;
struct _rwDlStateCache
{
    /* Z Buffer */
    RwBool                  zWriteEnable;
    RwBool                  zTestEnable;

#if !defined(NULL_GC)
    GXCompare               zCompare;
#endif /* !defined(NULL_GC) */

    /* Alpha blending */
    RwBool                  vertexAlpha;

    /* Culling mode */
    RwCullMode              cullMode;
#if !defined(NULL_GC)
    GXCullMode              gxCullMode;
#endif /* !defined(NULL_GC) */

    /* Fog */
    RwBool                  fogEnable;
    RwFogType               fogType;
    RwReal                  fogDensity;
    RwUInt32                packedFogColor;

#if !defined(NULL_GC)
    GXColor                 fogColor;
#endif /* !defined(NULL_GC) */

    /*RwReal                  fogFarPlane;*/

    /* Texturing stuff */
    RwRaster                *curTexRaster;
    RwTextureFilterMode     filterMode;
    RwTextureAddressMode    texAddressU;
    RwTextureAddressMode    texAddressV;
    RwBool                  rasterDirty;

#if !defined(NULL_GC)
    GXTexFilter             minTexFilter;
    GXTexFilter             maxTexFilter;
#endif /* !defined(NULL_GC) */

    /* SRC, DST blending modes */
    RwBlendFunction         srcBlend;
    RwBlendFunction         dstBlend;

#if !defined(NULL_GC)
    GXBlendFactor           gxSrcBlend;
    GXBlendFactor           gxDstBlend;
#endif /* !defined(NULL_GC) */

    /* Alpha comparison reference values */
    RwUInt8                 alphaCompRef0;
    RwUInt8                 alphaCompRef1;

    RwUInt8                 alphaEnabled;

    RwUInt8                 pad[2];
};

extern _rwDlStateCache    _RwDlStateCache;

/*----------------------------------------------------------------------*
 *-   Defines                                                          -*
 *----------------------------------------------------------------------*/

#define MATFXGCNENVMAPGETDATA(material)                                \
  &((*((rpMatFXMaterialData **)                                         \
       (((RwUInt8 *)material) +                                         \
        MatFXMaterialDataOffset)))->data[rpSECONDPASS].data.envMap)

#define MATFXGCNBUMPMAPGETDATA(material)                                \
  &((*((rpMatFXMaterialData **)                                         \
       (((RwUInt8 *)material) +                                         \
        MatFXMaterialDataOffset)))->data[rpSECONDPASS].data.bumpMap)

#define MATFXGCNDUALPASSGETDATA(material)                                \
  &((*((rpMatFXMaterialData **)                                         \
       (((RwUInt8 *)material) +                                         \
        MatFXMaterialDataOffset)))->data[rpSECONDPASS].data.dual)

#define MATFXGCNBUMPENVMAPGETDATA(material)                                \
  &((*((rpMatFXMaterialData **)                                         \
       (((RwUInt8 *)material) +                                         \
        MatFXMaterialDataOffset)))->data[rpTHIRDPASS].data.envMap)

#define RXGC_MAX_HW_LIGHTS         8

#define NUMADDRESSMODES 5

#define NUMBLENDMODES   12

#define NUMFILTERMODES  7

#define NUMBERSTAGES    8

#define LIGHTING_LARGE_NUMBER 1048576

/*----------------------------------------------------------------------*
 *-   Local/static Globals                                             -*
 *----------------------------------------------------------------------*/
static RxPipeline   *MatFXAtomicPipe = NULL,
					*MatFXWorldSectorPipe = NULL;

static GXLightID    _RpGCLightConvID[8] ATTRIBUTE_ALIGN(32) = {
	GX_LIGHT0,
	GX_LIGHT1,
	GX_LIGHT2,
	GX_LIGHT3,
	GX_LIGHT4,
	GX_LIGHT5,
	GX_LIGHT6,
	GX_LIGHT7
};

GXTexWrapMode RpGameCubeAddressConvTable[NUMADDRESSMODES] =
{
    (GXTexWrapMode)0,   /* rwTEXTUREADDRESSNATEXTUREADDRESS */
    GX_REPEAT,          /* rwTEXTUREADDRESSWRAP */
    GX_MIRROR,          /* rwTEXTUREADDRESSMIRROR */
    GX_CLAMP,           /* rwTEXTUREADDRESSCLAMP */
    (GXTexWrapMode)0    /* rwTEXTUREADDRESSBORDER - Not supported on Gamecube */
};

_RpGameCubeFilterParams RpGameCubeFilterModeConvTable[NUMFILTERMODES] =
{
    {(GXTexFilter)0,   (GXTexFilter)0}, /* rwFILTERNAFILTERMODE */
    {GX_NEAR,          GX_NEAR},        /* rwFILTERNEAREST */
    {GX_LINEAR,        GX_LINEAR},      /* rwFILTERLINEAR */
    {GX_NEAR_MIP_NEAR, GX_NEAR},        /* rwFILTERMIPNEAREST */
    {GX_LIN_MIP_NEAR,  GX_LINEAR},      /* rwFILTERMIPLINEAR */
    {GX_NEAR_MIP_LIN,  GX_NEAR},        /* rwFILTERLINEARMIPNEAREST */
    {GX_LIN_MIP_LIN,   GX_LINEAR}       /* rwFILTERLINEARMIPLINEAR */
};

GXBlendFactor RpGameCubeBlendConvTable[NUMBLENDMODES] =
{
    (GXBlendFactor)0,   /* rwBLENDNABLEND*/
    GX_BL_ZERO,         /* rwBLENDZERO */
    GX_BL_ONE,          /* rwBLENDONE */
    GX_BL_SRCCLR,       /* rwBLENDSRCCOLOR */
    GX_BL_INVSRCCLR,    /* rwBLENDINVSRCCOLOR */
    GX_BL_SRCALPHA,     /* rwBLENDSRCALPHA */
    GX_BL_INVSRCALPHA,  /* rwBLENDINVSRCALPHA */
    GX_BL_DSTALPHA,     /* rwBLENDDESTALPHA */
    GX_BL_INVDSTALPHA,  /* rwBLENDINVDESTALPHA */
    GX_BL_DSTCLR,       /* rwBLENDDESTCOLOR */
    GX_BL_INVDSTCLR,    /* rwBLENDINVDESTCOLOR */
    (GXBlendFactor)0    /* rwBLENDSRCALPHASAT - Not supported on Gamecube */
};

GXTlut RpGameCubeTlutConvTable[NUMBERSTAGES] = 
{
    GX_TLUT0,
    GX_TLUT1,
    GX_TLUT2,
    GX_TLUT3,
    GX_TLUT4,
    GX_TLUT5,
    GX_TLUT6,
    GX_TLUT7
};

GXTexMapID RpGameCubeTexMapIDConvTable[NUMBERSTAGES] = 
{
    GX_TEXMAP0,
    GX_TEXMAP1,
    GX_TEXMAP2,
    GX_TEXMAP3,
    GX_TEXMAP4,
    GX_TEXMAP5,
    GX_TEXMAP6,
    GX_TEXMAP7,
};

static GXLightObj   _RpGCLightObjs[8] ATTRIBUTE_ALIGN(32);


static GXColor  _RpGCWhite = {0xFF, 0xFF, 0xFF, 0xFF};
static GXColor  _RpGCBlack = {0x00, 0x00, 0x00, 0xFF};


/*----------------------------------------------------------------------*
 *-   Globals across program                                           -*
 *----------------------------------------------------------------------*/

/*----------------------------------------------------------------------*
 *-   Functions                                                        -*
 *----------------------------------------------------------------------*/
static void
_rpMatFxTextureRasterFlush(RwTexture* texture, RwInt32 stage)
{
    static GXTexObj texObj;
    RwInt32        format;
    RwRaster*      raster;
    RwGameCubeRasterExtension  *rasExt;

    RWFUNCTION(RWSTRING("_rpMatFxTextureRasterFlush"));

    if(texture!=NULL)
    {
        raster = texture->raster;
        format = RwRasterGetFormat(raster);
        rasExt = RwGameCubeRasterGetExtension(raster->parent);    
    }

    if ((rwRASTERFORMATPAL4 | rwRASTERFORMATPAL8) & format)
    {
        /*
         * Initialize the TLUT object
         */
        GXInitTexObjCI(&texObj,
                           (void *)rasExt->pixels,
                           (RwUInt16)raster->width, 
                           (RwUInt16)raster->height,
                           (GXCITexFmt)rasExt->format,
                           RpGameCubeAddressConvTable[texture->addressingU],
                           RpGameCubeAddressConvTable[texture->addressingV],
                           rasExt->mipmap,
                           RpGameCubeTlutConvTable[stage]);
            
        GXLoadTlut((GXTlutObj*)&rasExt->tlutObj, RpGameCubeTlutConvTable[stage]);
    }
    else
    {
        /*
         * Initialize the texture object
         */

        GXInitTexObj(&texObj,
                     (void *)rasExt->pixels,
                     (RwUInt16)raster->width, 
                     (RwUInt16)raster->height,
                     (GXTexFmt)rasExt->format,
                     RpGameCubeAddressConvTable[texture->addressingU],
                     RpGameCubeAddressConvTable[texture->addressingV],
                     rasExt->mipmap);
    }
    
    GXLoadTexObj(&texObj, RpGameCubeTexMapIDConvTable[stage]);

    if (1 == rasExt->alpha)
    {

         GXSetAlphaCompare(GX_GREATER, _RwDlStateCache.alphaCompRef0,
                           GX_AOP_AND,
                           GX_GREATER, _RwDlStateCache.alphaCompRef1);   

                /* Z Buffer after texturing */
         GXSetZCompLoc(GX_FALSE);
                
    }
    else
    {
         GXSetAlphaCompare(GX_ALWAYS, _RwDlStateCache.alphaCompRef0,
                           GX_AOP_AND,
                           GX_ALWAYS, _RwDlStateCache.alphaCompRef1);

         /* Z Buffer before texturing */
         GXSetZCompLoc(GX_TRUE);
    }
    RWRETURNVOID();
}


static void
_rpGCMatFxSetTexture(RwTexture* texture, RwInt32 flags, RwInt32 stage)
{
    RWFUNCTION(RWSTRING("_rpMatFxSetTexture"));

    if (((rxGEOMETRY_TEXTURED & flags) ||
         (rpGEOMETRYTEXTURED2 & flags))
        && (NULL != texture))
    {
        _rpMatFxTextureRasterFlush(texture, stage);
    }
    else
    {            
        _rpMatFxTextureRasterFlush(NULL, 0);
    }

    RWRETURNVOID();
}



static void
_rpGCMatFxTEVHWLightsSetup(RxGameCubePipeData *pipeData)
{
    RwInt32 flags;

    RWFUNCTION(RWSTRING("_rxGCTEVSetup"));

    flags = pipeData->flags;

    if (pipeData->dirLightMask || pipeData->spotLightMask)
    {
        /*
         * HardWare lights
         */

        if (rxGEOMETRY_PRELIT & flags)
        {
            /*
             * NO ambient light color.
             * NO modulation with material color.
             */

            /*
             * TEV - CHANNELS
             */

            /*
             * Set material color in GXSetChanCtrl() to white.
             * Set pre-light as ambient color in GXSetChanCtrl().
             */
            if (pipeData->dirLightMask)
            {
                GXSetChanCtrl(GX_COLOR0A0,      /* Color channel */
                              GX_TRUE,          /* Lights */
                              GX_SRC_VTX,       /* Ambient src pre-light */
                              GX_SRC_REG,       /* Material color is white of material color */
                              (GXLightID)pipeData->dirLightMask,    /* Lights */
                              GX_DF_CLAMP,       /* No diffuse function */
                              GX_AF_NONE);      /* No attenuation function */

                GXSetChanMatColor(GX_COLOR0A0, _RpGCWhite); /* Opaque white */
            }
        }
        else
        {
            /*
             * Ambient lighting.
             * HardWare lights.
             * Modulation with material color.
             */

            if (TRUE == pipeData->ambientLight)
            {
                /*
                 * TEV - CHANNELS 
                 */

                /*
                 * Set ambient color in GXSetChanCtrl() to accumulated ambient light.
                 */
                if (pipeData->dirLightMask)
                {
                    GXColor color;

                    GXSetChanCtrl(GX_COLOR0A0,      /* Color channel */
                                  GX_TRUE,          /* Lights */
                                  GX_SRC_REG,       /* Ambient src use the ambient lights */
                                  GX_SRC_REG,       /* Material color is white of material color */
                                  (GXLightID)pipeData->dirLightMask,    /* Lights */
                                  GX_DF_CLAMP,       /* No diffuse function */
                                  GX_AF_NONE);      /* No attenuation function */

                    color.r = (RwUInt8)(pipeData->ambientLightColor.red   * 255.0f);
                    color.g = (RwUInt8)(pipeData->ambientLightColor.green * 255.0f);
                    color.b = (RwUInt8)(pipeData->ambientLightColor.blue  * 255.0f);
                    color.a = 0xFF;

                    GXSetChanAmbColor(GX_COLOR0A0, color);
                }
            }
            else
            {
                /*
                 * TEV - CHANNELS 
                 */
                if (pipeData->dirLightMask)
                {
                    GXSetChanCtrl(GX_COLOR0A0,      /* Color channel */
                                  GX_TRUE,          /* Lights */
                                  GX_SRC_REG,       /* Ambient src use the ambient lights */
                                  GX_SRC_REG,       /* Material color is white of material color */
                                  (GXLightID)pipeData->dirLightMask,    /* Lights */
                                  GX_DF_CLAMP,       /* No diffuse function */
                                  GX_AF_NONE);      /* No attenuation function */

                    /*
                     * Set ambient color in GXSetChanCtrl() to black.
                     */
                    GXSetChanAmbColor(GX_COLOR0A0, _RpGCBlack); /* Opaque black */
                }
            }

            /*
             * If modulating set material color in GXSetChanCtrl() to
             * RpMaterial color per material else white.
             */
            if (!(rxGEOMETRY_MODULATE & flags))
            {
                GXSetChanMatColor(GX_COLOR0A0, _RpGCWhite); /* Opaque white */
            }
        }

        if ((pipeData->dirLightMask > 0) ^ (pipeData->spotLightMask > 0))
        {
            /*
             * Directional or Spot (exclusive).
             */

            /*
             * TEV - CHANNELS 
             */
            GXSetNumChans(1);

            /*
             * TEV - OPERATION
             */
        }
        else
        {
#if 0
            /*
             * Directional & Spot.
             */

            /*
             * TEV - CHANNELS 
             */
            GXSetNumChans(2);

            /*
             * TEV - OPERATION
             */
            GXSetNumTevStages(2);

            if (rxGEOMETRY_TEXTURED & flags)
            {
                /*
                 * Set TEV stage0 to combine color fragments.
                 * Set TEV stage1 to modulate the output of TEV stage0 with
                 * the texture.
                 */
            }
            else
            {
                /*
                 * Set TEV stage0 to combine color fragments.
                 */
            }
#endif /* 0/1 */
        }
    }
    else if ((TRUE == pipeData->ambientLight) && !(rxGEOMETRY_PRELIT & flags))
    {
        /*
         * NO HardWare lights.
         * NO pre-lighting.
         * Ambient light.
         */

        /*
         * TEV - CHANNELS 
         */
        GXSetNumChans(1);

        GXSetChanCtrl(GX_COLOR0A0,      /* Color channel */
                      GX_FALSE,         /* No lights */
                      GX_SRC_REG,       /* Ambient src not used */
                      GX_SRC_REG,       /* Material color use the ambient lights */
                      GX_LIGHT_NULL,    /* No lights */
                      GX_DF_NONE,       /* No diffuse function */
                      GX_AF_NONE);      /* No attenuation function */

    }
    else
    {
        if (rxGEOMETRY_PRELIT & flags)
        {
            /*
             * TEV - CHANNELS 
             */
            GXSetNumChans(1);

            GXSetChanCtrl(GX_COLOR0,        /* Color channel */
                          GX_FALSE,         /* No lights */
                          GX_SRC_REG,       /* Ambient src not used */
                          GX_SRC_VTX,       /* Material color use the prelight */
                          GX_LIGHT_NULL,    /* No lights */
                          GX_DF_NONE,       /* No diffuse function */
                          GX_AF_NONE);      /* No attenuation function */

            GXSetChanCtrl(GX_ALPHA0,        /* Color channel */
                          GX_FALSE,         /* No lights */
                          GX_SRC_REG,       /* Ambient src not used */
                          GX_SRC_REG,       /* Material color use opaque */
                          GX_LIGHT_NULL,    /* No lights */
                          GX_DF_NONE,       /* No diffuse function */
                          GX_AF_NONE);      /* No attenuation function */

            GXSetChanMatColor(GX_ALPHA0, _RpGCWhite); /* Opaque white */

        }
        else
        {
            /*
             * TEV - CHANNELS 
             */
            GXSetChanCtrl(GX_COLOR0A0,      /* Color channel */
                          GX_FALSE,         /* No lights */
                          GX_SRC_REG,       /* Ambient src not used */
                          GX_SRC_REG,       /* Material color use the prelight */
                          GX_LIGHT_NULL,    /* No lights */
                          GX_DF_NONE,       /* No diffuse function */
                          GX_AF_NONE);      /* No attenuation function */

            GXSetChanMatColor(GX_COLOR0A0, _RpGCBlack); /* Opaque black */
        }
    }

    RWRETURNVOID();
}

static void
_rpGCMatFxEnvMapTEVSetup(RpMaterial *material,
                         RxGameCubePipeData *pipeData,
                         RpAtomic *atomic)
{
    RwInt32         flags;
    RwMatrix        texMatrix, invCamMatrix, dp_matrix;
    RwV3d           v;
    Mtx             gxMtx;
    MatFXEnvMapData *envMapData;

    RWFUNCTION(RWSTRING("_rpGCMatFxTEVTexSetup"));

    flags = pipeData->flags;

    RwMatrixSetIdentity(&texMatrix);

    v.x =  0.5f;
    v.y = -0.5f;
    v.z =  0.0f;
    RwMatrixScale(&texMatrix, &v, rwCOMBINEPOSTCONCAT);

    v.x =  0.5f;
    v.y =  0.5f;
    v.z =  1.0f;
    RwMatrixTranslate(&texMatrix, &v, rwCOMBINEPOSTCONCAT);

    envMapData = MATFXGCNENVMAPGETDATA(material);
    if (envMapData->frame)
    {
        RwMatrix    invFrameLTM;

        RwMatrixSetIdentity(&invFrameLTM);

        /* Calc the inverse of the frames LTM */
        RwMatrixInvert(&invFrameLTM, RwFrameGetLTM(envMapData->frame));

        /* Calc normals transform matrix, world space */
        RwMatrixMultiply(&invCamMatrix, &invFrameLTM,
                         RwFrameGetLTM(RpAtomicGetFrame(atomic)));

        /* UV shift & scale */
        RwMatrixMultiply(&dp_matrix, &invCamMatrix, &texMatrix);
    }
    else
    {
        RwMatrixMultiply(&dp_matrix,
                         RwFrameGetLTM(RpAtomicGetFrame(atomic)),
                         &texMatrix);
    }

    gxMtx[0][0] = -dp_matrix.right.x;
    gxMtx[0][1] = -dp_matrix.up.x;
    gxMtx[0][2] = -dp_matrix.at.x;
    gxMtx[0][3] = -dp_matrix.pos.x;

    gxMtx[1][0] =  dp_matrix.right.y;
    gxMtx[1][1] =  dp_matrix.up.y;
    gxMtx[1][2] =  dp_matrix.at.y;
    gxMtx[1][3] =  dp_matrix.pos.y;

    gxMtx[2][0] = -dp_matrix.right.z;
    gxMtx[2][1] = -dp_matrix.up.z;
    gxMtx[2][2] = -dp_matrix.at.z;
    gxMtx[2][3] = -dp_matrix.pos.z;

    GXLoadTexMtxImm(gxMtx, GX_TEXMTX0, GX_MTX3x4);


    GXSetNumTevStages(2);

#ifdef EMU
    if ((flags & rxGEOMETRY_TEXTURED))
    {
        /* GX_MODULATE - modulate with rasterized color */
        GXSetTevOp(GX_TEVSTAGE0, GX_MODULATE);
        GXSetTevOp(GX_TEVSTAGE1, GX_BLEND);

        /* UVs */
        GXSetNumTexGens(2);
        GXSetTexCoordGen(GX_TEXCOORD0, GX_TG_MTX2x4, GX_TG_TEX0, GX_IDENTITY);
        GXSetTexCoordGen(GX_TEXCOORD1, GX_TG_MTX3x4, GX_TG_NRM, GX_TEXMTX0);

        GXSetTevOrder(GX_TEVSTAGE0, GX_TEXCOORD0, GX_TEXMAP0, GX_COLOR0A0);
        GXSetTevOrder(GX_TEVSTAGE1, GX_TEXCOORD1, GX_TEXMAP1, GX_COLOR0A0);
    }
    else
    {
        /* GX_PASSCLR - rasterized color only */
        GXSetTevOp(GX_TEVSTAGE0, GX_PASSCLR);

        /* No UVs */
        GXSetNumTexGens(0);
		GXSetTevOrder(GX_TEVSTAGE0, GX_TEXCOORD_NULL,
                      GX_TEXMAP_NULL, GX_COLOR0A0);
    }
#else
    if ((flags & rxGEOMETRY_TEXTURED))
    {
        GXColor shininess = { 255, 255, 255, 255 };
        /* UVs */
        GXSetNumTexGens(2);
        
        GXSetTexCoordGen(GX_TEXCOORD0, GX_TG_MTX2x4, GX_TG_TEX0, GX_IDENTITY);
        GXSetTexCoordGen(GX_TEXCOORD1, GX_TG_MTX3x4, GX_TG_NRM, GX_TEXMTX0);
        
        GXSetTevOrder( GX_TEVSTAGE0, GX_TEXCOORD0, GX_TEXMAP0, GX_COLOR0A0   );
        GXSetTevOrder( GX_TEVSTAGE1, GX_TEXCOORD1, GX_TEXMAP1, GX_COLOR_NULL );        

        GXSetTevColorIn( GX_TEVSTAGE0, GX_CC_ZERO,
                                       GX_CC_TEXC,  
                                       GX_CC_RASC,  
                                       GX_CC_ZERO   );
        GXSetTevColorOp( GX_TEVSTAGE0, GX_TEV_ADD,
                                       GX_TB_ZERO,
                                       GX_CS_SCALE_1,
                                       GX_DISABLE,
                                       GX_TEVPREV   );

        GXSetTevAlphaIn( GX_TEVSTAGE0, GX_CA_ZERO, GX_CA_ZERO,
                                       GX_CA_ZERO, GX_CA_ONE );
        GXSetTevAlphaOp( GX_TEVSTAGE0, GX_TEV_ADD, GX_TB_ZERO,
                                       GX_CS_SCALE_1, GX_DISABLE, GX_TEVPREV );
                                       
        /*
         * Get shininess and set TEV color
         */                               
         shininess.a = (u8)(envMapData->coef * 255.0f);
         GXSetTevColor( GX_TEVREG0, shininess );

        GXSetTevColorIn( GX_TEVSTAGE1, GX_CC_ZERO,
                                       GX_CC_TEXC,  
                                       GX_CC_A0,    
                                       GX_CC_CPREV   );
        GXSetTevColorOp( GX_TEVSTAGE1, GX_TEV_ADD,  
                                       GX_TB_ZERO,
                                       GX_CS_SCALE_1,
                                       GX_ENABLE,
                                       GX_TEVPREV                        );

        GXSetTevAlphaIn( GX_TEVSTAGE1, GX_CA_ZERO, GX_CA_ZERO,
                                       GX_CA_ZERO, GX_CA_ONE );
        GXSetTevAlphaOp( GX_TEVSTAGE1, GX_TEV_ADD, GX_TB_ZERO,
                                       GX_CS_SCALE_1, GX_DISABLE, GX_TEVPREV );
        
    }
    else
    {
        /* GX_PASSCLR - rasterized color only */
        GXSetTevOp(GX_TEVSTAGE0, GX_PASSCLR);

        /* No UVs */
        GXSetNumTexGens(0);
		GXSetTevOrder(GX_TEVSTAGE0, GX_TEXCOORD_NULL,
                      GX_TEXMAP_NULL, GX_COLOR0A0);
    }

#endif


    RWRETURNVOID();
}


static void
_rpGCMatFxBumpMapTEVSetup(RxGameCubePipeData *pipeData)
{
    RwInt32 flags;
    /*GXColor bumpScale;*/

    RWFUNCTION(RWSTRING("_rpGCMatFxBumpMapTEVTexSetup"));

    flags = pipeData->flags;

    GXSetNumTevStages(2);

#if 1
    if ((flags & rxGEOMETRY_TEXTURED) || (flags & rpGEOMETRYTEXTURED2))
    {
        /* GX_MODULATE - modulate with rasterized color */
        GXSetTevOp(GX_TEVSTAGE0, GX_MODULATE);
        GXSetTevOp(GX_TEVSTAGE1, GX_DECAL);

        /* UVs */
        GXSetNumTexGens(2);
        GXSetTexCoordGen(GX_TEXCOORD0, GX_TG_MTX2x4, GX_TG_TEX0, GX_IDENTITY);
        GXSetTexCoordGen(GX_TEXCOORD1, GX_TG_MTX2x4, GX_TG_TEX0, GX_IDENTITY);

        GXSetTevOrder(GX_TEVSTAGE0, GX_TEXCOORD0, GX_TEXMAP0, GX_COLOR0A0);
        GXSetTevOrder(GX_TEVSTAGE1, GX_TEXCOORD0, GX_TEXMAP1, GX_COLOR0A0);
    }
    else
    {
        /* GX_PASSCLR - rasterized color only */
        GXSetTevOp(GX_TEVSTAGE0, GX_PASSCLR);

        /* No UVs */
        GXSetNumTexGens(0);
		GXSetTevOrder(GX_TEVSTAGE0, GX_TEXCOORD_NULL,
                      GX_TEXMAP_NULL, GX_COLOR0A0);
    }
#else
    if ((flags & rxGEOMETRY_TEXTURED) || (flags & rpGEOMETRYTEXTURED2))
    {
        //  Tex Gen
        //     GX_TEXCOORD0: Base texcoord of bump mapping 
        //     GX_TEXCOORD1: Tex gen by bump mapping
        GXSetNumTexGens(2); // coord1 is always the max coord
        GXSetTexCoordGen( GX_TEXCOORD0, GX_TG_MTX2x4, GX_TG_TEXCOORD0, GX_IDENTITY );
        GXSetTexCoordGen( GX_TEXCOORD1, GX_TG_BUMP0, GX_TG_TEXCOORD0, GX_IDENTITY );
    
        //  Tev Order
        //     GX_TEVSTAGE0: send base    texture + diffuse color
        //     GX_TEVSTAGE1: send shifted texture
        GXSetTevOrder( GX_TEVSTAGE0, GX_TEXCOORD0, GX_TEXMAP0, GX_COLOR0A0   );
        GXSetTevOrder( GX_TEVSTAGE1, GX_TEXCOORD1, GX_TEXMAP0, GX_COLOR_NULL );
        
        //  Tev Register
        //     GX_TEVREG0: ( X, X, X, bumpScale )
        bumpScale.a = 128;
        GXSetTevColor( GX_TEVREG0, bumpScale );

        //
        //  In many case, bump mapping needs 3 tev stage, but here,
        //  use only 2 tev stages, since no material texture map
        //  
        GXSetNumTevStages( 2 );
        
        //
        //  Tev stage0
        //     Color:   BumpTexture * bumpScale + DiffuseColor => R
        //     Alpha:   Not used
        //
        GXSetTevColorIn( GX_TEVSTAGE0, GX_CC_ZERO,
                                       GX_CC_TEXC,  /* Bump Texture   */
                                       GX_CC_A0,    /* BumpScale      */
                                       GX_CC_RASC   /* Diffuse color  */ );
        GXSetTevColorOp( GX_TEVSTAGE0, GX_TEV_ADD,
                                       GX_TB_ZERO,
                                       GX_CS_SCALE_1,
                                       GX_DISABLE,
                                       GX_TEVPREV   /* To next stage  */ );

        GXSetTevAlphaIn( GX_TEVSTAGE0, GX_CA_ZERO, GX_CA_ZERO,
                                       GX_CA_ZERO, GX_CA_ZERO );
        GXSetTevAlphaOp( GX_TEVSTAGE0, GX_TEV_ADD, GX_TB_ZERO,
                                       GX_CS_SCALE_1, GX_DISABLE, GX_TEVPREV );

        //
        //  Tev stage1
        //     Color:   ShiftTexture * bumpScale * (-1) + R  =>  R
        //     Alpha:   Not used
        //
        GXSetTevColorIn( GX_TEVSTAGE1, GX_CC_ZERO,
                                       GX_CC_TEXC,  /* Shift Texture  */
                                       GX_CC_A0,    /* BumpScale      */
                                       GX_CC_CPREV  /* Last stage out */ );
        GXSetTevColorOp( GX_TEVSTAGE1, GX_TEV_SUB,  /* mult (-1)      */
                                       GX_TB_ZERO,
                                       GX_CS_SCALE_1,
                                       GX_ENABLE,
                                       GX_TEVPREV                        );

        GXSetTevAlphaIn( GX_TEVSTAGE1, GX_CA_ZERO, GX_CA_ZERO,
                                       GX_CA_ZERO, GX_CA_ZERO );
        GXSetTevAlphaOp( GX_TEVSTAGE1, GX_TEV_ADD, GX_TB_ZERO,
                                       GX_CS_SCALE_1, GX_DISABLE, GX_TEVPREV );
    }
    else
    {
        /* GX_PASSCLR - rasterized color only */
        GXSetTevOp(GX_TEVSTAGE0, GX_PASSCLR);

        /* No UVs */
        GXSetNumTexGens(0);
		GXSetTevOrder(GX_TEVSTAGE0, GX_TEXCOORD_NULL,
                      GX_TEXMAP_NULL, GX_COLOR0A0);
    }
#endif

    RWRETURNVOID();
}

static void
_rpGCMatFxBumpEnvMapTEVSetup(RxGameCubePipeData *pipeData)
{
    RwInt32 flags;

    RWFUNCTION(RWSTRING("_rpGCMatFxTEVTexSetup"));

    flags = pipeData->flags;

    /*
     * TEV - OPERATION
     */
    GXSetNumTevStages(1);

    if ((rxGEOMETRY_TEXTURED & flags) ||
        (rpGEOMETRYTEXTURED2 & flags))
    {
        /* UVs */
        GXSetNumTexGens(1);
        GXSetTexCoordGen(GX_TEXCOORD0, GX_TG_MTX2x4, GX_TG_TEX0, GX_IDENTITY);

        /*
         * If modulating, modulate ambient color with material color per
         * material and set as material color in GXSetChanCtrl() else just
         * set material color per material.
         */

        /* GXSetTevOp is a convenience function, see docs */
        GXSetTevOp(GX_TEVSTAGE0, GX_MODULATE);
        GXSetTevOrder(GX_TEVSTAGE0, GX_TEXCOORD0, GX_TEXMAP0, GX_COLOR0A0);
    }
    else
    {
        GXSetNumTexGens(0);

        /*
         * If modulating, modulate ambient color with material color per
         * material and set as material color in GXSetChanCtrl() else just
         * set material color per material.
         */

        /* GXSetTevOp is a convenience function, see docs */
        GXSetTevOp(GX_TEVSTAGE0, GX_PASSCLR);
        GXSetTevOrder(GX_TEVSTAGE0, GX_TEXCOORD_NULL, GX_TEXMAP_NULL, GX_COLOR0A0);
    }

    RWRETURNVOID();
}

static void
_rpGCMatFxDualPassTEVSetup(RxGameCubePipeData *pipeData)
{
    RwInt32 flags;

    RWFUNCTION(RWSTRING("_rpGCMatFxDualPassTEVTexSetup"));

    flags = pipeData->flags;

    GXSetNumTevStages(2);

    if ((flags & rxGEOMETRY_TEXTURED) || (flags & rpGEOMETRYTEXTURED2))
    {
        /* GX_MODULATE - modulate with rasterized color */
        GXSetTevOp(GX_TEVSTAGE0, GX_MODULATE);
        GXSetTevOp(GX_TEVSTAGE1, GX_DECAL);

        /* UVs */
        GXSetNumTexGens(2);
        GXSetTexCoordGen(GX_TEXCOORD0, GX_TG_MTX2x4, GX_TG_TEX0, GX_IDENTITY);
        GXSetTexCoordGen(GX_TEXCOORD1, GX_TG_MTX2x4, GX_TG_TEX1, GX_IDENTITY);

        GXSetTevOrder(GX_TEVSTAGE0, GX_TEXCOORD0, GX_TEXMAP0, GX_COLOR0A0);
        GXSetTevOrder(GX_TEVSTAGE1, GX_TEXCOORD1, GX_TEXMAP1, GX_COLOR0A0);
    }
    else
    {
        /* GX_PASSCLR - rasterized color only */
        GXSetTevOp(GX_TEVSTAGE0, GX_PASSCLR);

        /* No UVs */
        GXSetNumTexGens(0);
		GXSetTevOrder(GX_TEVSTAGE0, GX_TEXCOORD_NULL,
                      GX_TEXMAP_NULL, GX_COLOR0A0);
    }


    RWRETURNVOID();
}



/*****************************************************************************
 _rpDlHWLightingApplyDirectionalLight

 On entry   : Instanced data
            : light
            : Optional inverse object matrix (to transform light to object space)
 On exit    :
 */
static void
_rpGCMatFxHWLightingApplyDirectionalLight(const RpLight *light, RwInt32 lightNum)
{
    RwMatrix            tmp;
    RwMatrix            matrix;
	const RwRGBAReal    *lightColor;
	GXColor             color;

    RWFUNCTION(RWSTRING("_rpGCHWLightingApplyDirectionalLight"));
    RWASSERT(light);

	/* Calculate the transform matrix */
    RwMatrixSetIdentity(&matrix);
    RwMatrixSetIdentity(&tmp);

    RwMatrixInvert(&tmp,
                   RwFrameGetLTM(RwCameraGetFrame(RWSRCGLOBAL(curCamera))));
    RwMatrixMultiply(&matrix, RwFrameGetLTM(RpLightGetFrame(light)), &tmp);

	GXInitLightPos(&_RpGCLightObjs[lightNum],
                   -matrix.at.x * -LIGHTING_LARGE_NUMBER,
                    matrix.at.y * -LIGHTING_LARGE_NUMBER,
                   -matrix.at.z * -LIGHTING_LARGE_NUMBER);

	GXInitLightAttn(&_RpGCLightObjs[lightNum],
                    1.0f,//material->surfaceProps.diffuse,
					0.0f,
					0.0f,
					1.0f,
					0.0f,
					0.0f);

	lightColor = RpLightGetColor(light);
	color.r = (RwUInt8)(lightColor->red   * 255.0f);
	color.g = (RwUInt8)(lightColor->green * 255.0f);
	color.b = (RwUInt8)(lightColor->blue  * 255.0f);
	color.a = 0xFF;

	GXInitLightColor(&_RpGCLightObjs[lightNum], color);
	GXLoadLightObjImm(&_RpGCLightObjs[lightNum], _RpGCLightConvID[lightNum]);

    RWRETURNVOID();
}


static void
_rpGCMatFxLightsGlobalEnable(RxGameCubePipeData *pipeData, RpLightFlag lightFlags)
{
    RpWorld     *world;
    RwLLLink    *cur, *end;

    RWFUNCTION(RWSTRING("_rwGCLightsGlobalEnable"));

    world = (RpWorld *)RWSRCGLOBAL(curWorld);

    cur = rwLinkListGetFirstLLLink(&world->directionalLightList);
    end = rwLinkListGetTerminator(&world->directionalLightList);

    while (cur != end)
    {
        const RwRGBAReal    *color;
        RpLight             *light;

        light = rwLLLinkGetData(cur, RpLight, inWorld);

        /* NB light may actually be a dummyTie from a enclosing ForAll */
        if (light && (rwObjectTestFlags(light, lightFlags)))
        {
            if (rpLIGHTDIRECTIONAL == RpLightGetType(light))
            {
                RWASSERT(pipeData->numLights <= RXGC_MAX_HW_LIGHTS);

                /* Setup light */
                _rpGCMatFxHWLightingApplyDirectionalLight(light,
                                                     pipeData->numLights);
                pipeData->dirLightMask |= _RpGCLightConvID[pipeData->numLights];
                pipeData->numLights++;
            }
            else
            {
                color = RpLightGetColor(light);

                /*pipeData->ambientLightColor.red   += color->red;
                pipeData->ambientLightColor.green += color->green;
                pipeData->ambientLightColor.blue  += color->blue;*/

                pipeData->ambientLightColor.red   += 0.5f;
                pipeData->ambientLightColor.green += 0.5f;
                pipeData->ambientLightColor.blue  += 0.5f;


                pipeData->ambientLight = TRUE;
            }
        }

        /* Next */
        cur = rwLLLinkGetNext(cur);
    }

    RWRETURNVOID();
}


static void *
_rpGCMatFxAtomicLightingCallback(void *object, RxGameCubePipeData *pipeData)
{
    RpAtomic	*atomic;
    RwInt32		flags;

    RWFUNCTION(RWSTRING("_rxGCAtomicDefaultLightingCallback"));

    atomic = (RpAtomic *)object;
    flags = RpGeometryGetFlags(RpAtomicGetGeometry(atomic));

    pipeData->dirLightMask = 0;
    pipeData->spotLightMask = 0;
    pipeData->ambientLightColor.red = 0.0f;
    pipeData->ambientLightColor.green = 0.0f;
    pipeData->ambientLightColor.blue = 0.0f;
    pipeData->ambientLightColor.alpha = 1.0f;
    pipeData->ambientLight = FALSE;
    pipeData->numLights = 0;

    if ((rxGEOMETRY_LIGHT & flags) && (NULL != RWSRCGLOBAL(curWorld)))
    {
        RwLLLink    *cur, *end;
        RwUInt32    numLights = 0;
        RwBool      lighting = FALSE;

        /*
         * Global lights, (Directional & Ambient)
         */
       _rpGCMatFxLightsGlobalEnable(pipeData, rpLIGHTLIGHTATOMICS);

        /*
         * Local Lights, (Point, Soft & Soft spot)
         */

        /* Increase the marker ! */
        RWSRCGLOBAL(lightFrame)++;

        /* For all sectors that this atomic lies in, apply all lights within */
        cur = rwLinkListGetFirstLLLink(&atomic->llWorldSectorsInAtomic);
        end = rwLinkListGetTerminator(&atomic->llWorldSectorsInAtomic);

        while (cur != end)
        {
            RpTie           *tpTie = rwLLLinkGetData(cur, RpTie, lWorldSectorInAtomic);
            RwLLLink        *curLight, *endLight;

            /* Lights in the sector */
            curLight = rwLinkListGetFirstLLLink(&tpTie->worldSector->lightsInWorldSector);
            endLight = rwLinkListGetTerminator(&tpTie->worldSector->lightsInWorldSector);

            while (curLight != endLight)
            {
                RpLight     *light;
                RpLightTie  *lightTie;
                //const RwRGBAReal    *color;

                lightTie = rwLLLinkGetData(curLight, RpLightTie, lightInWorldSector);
                light = lightTie->light;

                /* NB lightTie may actually be a dummyTie from a enclosing ForAll */

                /* Check to see if the light is set to light worlds */
                if (light && (rwObjectTestFlags(light, rpLIGHTLIGHTWORLD)))
                {
                    switch (RpLightGetType(light))
                    {
                        case rpLIGHTPOINT:
                        case rpLIGHTSPOT:
                        case rpLIGHTSPOTSOFT:
                        {
                            //RWASSERT(pipeData->numDirLights + pipeData->numSpotLights < RXGC_MAX_HW_LIGHTS));
                            //color = RpLightGetColor(light);
                            break;
                        }

                        case rpLIGHTAMBIENT:
                        case rpLIGHTDIRECTIONAL:
                        default:
                        {
                            break;
                        }
                    }
                }

                /* Next */
                curLight = rwLLLinkGetNext(curLight);
            }

            /* Next one */
            cur = rwLLLinkGetNext(cur);
        }
    }

    if (pipeData->ambientLight)
    {
        if (pipeData->ambientLightColor.red > 1.0f)
	    {
            pipeData->ambientLightColor.red = 1.0f;
        }
        if (pipeData->ambientLightColor.green > 1.0f)
	    {
            pipeData->ambientLightColor.green = 1.0f;
	    }
        if (pipeData->ambientLightColor.blue > 1.0f)
	    {
            pipeData->ambientLightColor.blue = 1.0f;
	    }
    }

    RWRETURN(object);
}

static void
_rpComputeBinormalsAndTangents(RwV3d *binormals,
                               RwV3d *tangents,
                               RpAtomic *object,
                               RxGameCubeInstanceData *instanceData,
                               RxGameCubeResEntryHeader *resEntryHeader,
                               MatFXBumpMapData *bumpMapData)
{
    RwInt32     i;    
    RwFrame *bumpFrame  = bumpMapData->frame;
    RwV3d lightPosObj, *lightPos, l, temp1, temp2, b, t;
    RwMatrix worldToObj;
    RwMatrix *transform;    

    RWFUNCTION(RWSTRING("_rpComputeBinormalsAndTangents"));

    if (bumpFrame == NULL)
    {
        bumpFrame = RwCameraGetFrame(RwCameraGetCurrentCamera());
        RWASSERT(bumpFrame);
    }

    lightPos = RwMatrixGetPos(RwFrameGetLTM(bumpFrame));

    transform = RwFrameGetLTM(RpAtomicGetFrame(object));

    RwMatrixInvert(&worldToObj, transform);
    RwV3dTransformPoints(&lightPosObj, lightPos, 1, &worldToObj);

    for( i = 0; i < object->geometry->numTriangles ; i++ )
    {
        RwReal unused;
        RwReal vv1, vv2, vv3;
        RwV3d *n1, *n2, *n3;
        RwV3d *v1, *v2, *v3;
        RwUInt16 *indices;

        indices =(RwUInt16*) &((RpAtomic*)object)->geometry->triangles[i].vertIndex;
        v1 = (RwV3d*) &((RwV3d*)resEntryHeader->positions)[indices[0]];
        v2 = (RwV3d*) &((RwV3d*)resEntryHeader->positions)[indices[1]];
        v3 = (RwV3d*) &((RwV3d*)resEntryHeader->positions)[indices[2]];

        n1 = (RwV3d*) &((RwV3d*)resEntryHeader->normals)[indices[0]];
        n2 = (RwV3d*) &((RwV3d*)resEntryHeader->normals)[indices[1]];
        n3 = (RwV3d*) &((RwV3d*)resEntryHeader->normals)[indices[2]];

        RwV3dSub(&l, &lightPosObj, v1);
        RwV3dNormalize(&l, &l);

        RwV3dSub(&temp1, v2, v1);
        RwV3dSub(&temp2, v3, v1);

        vv1 = ((RwV2d*)resEntryHeader->texCoords[0])[indices[0]].y;
        vv2 = ((RwV2d*)resEntryHeader->texCoords[0])[indices[1]].y;
        vv3 = ((RwV2d*)resEntryHeader->texCoords[0])[indices[2]].y;

        RwV3dScale(&temp1, &temp1, vv3 - vv1);
        RwV3dScale(&temp2, &temp2, vv2 - vv1);

        RwV3dSub(&t, &temp1, &temp2);
        _rwV3dNormalizeMacro(unused, &t, &t);
        RwV3dCrossProduct(&b, &t, n1);

        binormals[indices[0]]=b;
        tangents[indices[0]]=t;

        RwV3dScale(&temp1, &temp1, vv1 - vv2);
        RwV3dScale(&temp2, &temp2, vv3 - vv2);

        RwV3dSub(&t, &temp1, &temp2);
        _rwV3dNormalizeMacro(unused, &t, &t);
        RwV3dCrossProduct(&b, &t, n2);

        binormals[indices[1]]=b;
        tangents[indices[1]]=t;

        RwV3dScale(&temp1, &temp1, vv1 - vv3);
        RwV3dScale(&temp2, &temp2, vv2 - vv3);

        RwV3dSub(&t, &temp1, &temp2);
        _rwV3dNormalizeMacro(unused, &t, &t);
        RwV3dCrossProduct(&b, &t, n3);

        binormals[indices[2]]=b;
        tangents[indices[2]]=t;
    }

    RWRETURNVOID();
}

/*
 * Specialised render callback for Bump mapping
 */
static void *
_rpGCMatFxAtomicDirectRender(void *object, RxGameCubePipeData *pipeData,
                       RxGameCubeInstanceData *instancedData,
                       RxGameCubeResEntryHeader *resEntryHeader,
                       MatFXBumpMapData *bumpMapData)
{
    RwV3d               *up, *at, *right, *pos;
    RwInt32             flags;
    RwInt32             i, numVertices;
    RwUInt16            *indices;
    RwMatrix            transformMatrix;
    Mtx                 mtx;
    RwV3d*              binormals;
    RwV3d*              tangents;

    RWFUNCTION(RWSTRING("_rpGCMatFxRenderCallback"));

    /* 
     * First we calculate the binormals and tangents
     */
     binormals = (RwV3d*)RwMalloc( ((RpAtomic*)object)->geometry->numTriangles * 3 * sizeof(RwV3d));
     tangents = (RwV3d*)RwMalloc( ((RpAtomic*)object)->geometry->numTriangles * 3 * sizeof(RwV3d));

     _rpComputeBinormalsAndTangents(binormals, 
                                    tangents, 
                                    (RpAtomic*)object, 
                                    instancedData,
                                    resEntryHeader,
                                    bumpMapData );

    /*
     * Calculate the transform matrix.
     */
    if (rpATOMIC == RwObjectGetType(object))
    {
        /* Calculate the transform matrix */
        RwMatrix tmp;

        RwMatrixSetIdentity(&transformMatrix);
        RwMatrixSetIdentity(&tmp);

        RwMatrixInvert(&tmp, RwFrameGetLTM(RwCameraGetFrame(RWSRCGLOBAL(curCamera))));
        RwMatrixMultiply(&transformMatrix,
                         RwFrameGetLTM(RpAtomicGetFrame((RpAtomic *)object)),
                         &tmp);
    }
    else
    {
        RwMatrixInvert(&transformMatrix,
                   RwFrameGetLTM(RwCameraGetFrame(RWSRCGLOBAL(curCamera))));
    }

    /*
	 * Set the transform matrix
	 */
    right = &transformMatrix.right;
    up = &transformMatrix.up;
    at = &transformMatrix.at;
    pos = &transformMatrix.pos;

    mtx[0][0] = -right->x;
    mtx[0][1] = -up->x;
    mtx[0][2] = -at->x;
    mtx[0][3] = -pos->x;

    mtx[1][0] = right->y;
    mtx[1][1] = up->y;
    mtx[1][2] = at->y;
    mtx[1][3] = pos->y;

    mtx[2][0] = -right->z;
    mtx[2][1] = -up->z;
    mtx[2][2] = -at->z;
    mtx[2][3] = -pos->z;

    GXLoadPosMtxImm(mtx, GX_PNMTX0);

    flags = pipeData->flags;

    if (rxGEOMETRY_NORMALS & flags)
	{
        GXLoadNrmMtxImm(mtx, GX_PNMTX0);
	}

    /* Use GX_PNMTX0 as the default matrix ID */
    GXSetCurrentMtx(GX_PNMTX0);

    /*
     * Setup the lights
     */
    //_rxGCHWLightsSetup(object, pipeData);


    /*
     * Vertex setup
     */

    /* Set vertex descriptor to GX_NONE */
    GXClearVtxDesc();

    /* Position */
    GXSetVtxDesc(GX_VA_POS, GX_INDEX16); /* 16-bit indices */
    GXSetVtxAttrFmt(GX_VTXFMT0,
                    GX_VA_POS,
                    GX_POS_XYZ,
                    GX_F32,
                    0);
    RWASSERT(NULL != resEntryHeader->positions);
    GXSetArray(GX_VA_POS, (void *)(resEntryHeader->positions), sizeof(RwV3d));

#if 1
    /* Normals */
    if ((rxGEOMETRY_NORMALS & flags) && (rxGEOMETRY_LIGHT & flags))
    {
        GXSetVtxDesc(GX_VA_NRM, GX_INDEX16); /* 16-bit indices */
        GXSetVtxAttrFmt(GX_VTXFMT0,
                        GX_VA_NRM,
                        GX_NRM_XYZ,
                        GX_F32,
                        0);
        RWASSERT(NULL != resEntryHeader->normals);
        GXSetArray(GX_VA_NRM, (void *)(resEntryHeader->normals), sizeof(RwV3d));
    }
#endif

    /* Pre-lighting */
    if (rxGEOMETRY_PRELIT & flags)
    {
        /* Color */
        GXSetVtxDesc(GX_VA_CLR0, GX_INDEX16); /* 16-bit indices */
        GXSetVtxAttrFmt(GX_VTXFMT0,
                        GX_VA_CLR0,
                        GX_CLR_RGB,
                        GX_RGB8,
                        0);
        RWASSERT(NULL != resEntryHeader->preLight);
        GXSetArray(GX_VA_CLR0, (void *)(resEntryHeader->preLight), sizeof(RwRGBA));
    }

    /* Texture coordinates */
    if (rxGEOMETRY_TEXTURED & flags)        
    {
        /* UVs */
        GXSetVtxDesc(GX_VA_TEX0, GX_INDEX16); /* 16-bit indices */
        GXSetVtxAttrFmt(GX_VTXFMT0,
                        GX_VA_TEX0,
                        GX_TEX_ST,
                        GX_F32,
                        0);
        RWASSERT(NULL != resEntryHeader->texCoords[0]);
        GXSetArray(GX_VA_TEX0, (void *)(resEntryHeader->texCoords[0]), sizeof(RwTexCoords));
    }

    /* Texture coordinates */
    if (rpGEOMETRYTEXTURED2 & flags)        
    {
        /* UVs */
        GXSetVtxDesc(GX_VA_TEX0, GX_INDEX16); /* 16-bit indices */
        GXSetVtxAttrFmt(GX_VTXFMT0,
                        GX_VA_TEX0,
                        GX_TEX_ST,
                        GX_F32,
                        0);
        RWASSERT(NULL != resEntryHeader->texCoords[0]);
        GXSetArray(GX_VA_TEX0, (void *)(resEntryHeader->texCoords[0]), sizeof(RwTexCoords));

        /* UVs */
        GXSetVtxDesc(GX_VA_TEX1, GX_INDEX16); /* 16-bit indices */
        GXSetVtxAttrFmt(GX_VTXFMT0,
                        GX_VA_TEX1,
                        GX_TEX_ST,
                        GX_F32,
                        0);
        RWASSERT(NULL != resEntryHeader->texCoords[1]);
        GXSetArray(GX_VA_TEX1, (void *)(resEntryHeader->texCoords[1]), sizeof(RwTexCoords));
    }


#if 0
    /* Binormals */
    GXSetVtxDesc(GX_VA_NBT, GX_DIRECT);
    GXSetVtxAttrFmt(GX_VTXFMT0,
                    GX_VA_NBT,
                    GX_NRM_NBT3,
                    GX_F32,
                    0 );
#endif

    /*
     * Renderstates
     */

    if (!(rxGEOMETRY_PRELIT & pipeData->flags) &&
         (rxGEOMETRY_MODULATE & pipeData->flags))
    {
        if (pipeData->numLights > 0)
        {
            /*
             * Set material color in GXSetChanCtrl() to RpMaterial color.
             */
            GXSetChanMatColor(GX_COLOR0A0,
                              *(GXColor *)&instancedData->material->color);
        }
        else if (TRUE == pipeData->ambientLight)
        {
            GXColor color;

            /*
             * Modulate ambient color with material color per material and set
             * as material color in GXSetChanCtrl().
             */
            color.r = (RwUInt8)((RwReal)instancedData->material->color.red *
                                pipeData->ambientLightColor.red);
            color.g = (RwUInt8)((RwReal)instancedData->material->color.green *
                                pipeData->ambientLightColor.green);
            color.b = (RwUInt8)((RwReal)instancedData->material->color.blue *
                                pipeData->ambientLightColor.blue);
            color.a = instancedData->material->color.alpha;

            GXSetChanMatColor(GX_COLOR0A0, color);
        }        
    }

    /*
     * Send the datas directly
     */

    numVertices = ((RpAtomic*)object)->geometry->numTriangles * 3;
    GXBegin( GX_TRIANGLES, GX_VTXFMT0, numVertices );

    // Send the vertices data one by one
    for( i = 0 ; i < (numVertices/3) ; i++ )
    {
        indices =(RwUInt16*) &((RpAtomic*)object)->geometry->triangles[i].vertIndex;        

        /* 1st vertex */
        GXPosition1x16(indices[0]);     /* Position */
#if 1
        GXNormal1x16(indices[0]);       /* Normal */
#endif
        if (rxGEOMETRY_PRELIT & flags) GXColor1x16(indices[0]);
        if (rxGEOMETRY_TEXTURED & flags) GXTexCoord1x16(indices[0]);
        if (rpGEOMETRYTEXTURED2 & flags) GXTexCoord1x16(indices[0]);
      
#if 0
        GXNormal3f32(((RwV3d*)resEntryHeader->normals)[indices[0]].x,
                     ((RwV3d*)resEntryHeader->normals)[indices[0]].y,
                     ((RwV3d*)resEntryHeader->normals)[indices[0]].z);

        GXNormal3f32(binormals[indices[0]].x,
                     binormals[indices[0]].y,
                     binormals[indices[0]].z );
                     
        GXNormal3f32(tangents[indices[0]].x,
                     tangents[indices[0]].y,
                     tangents[indices[0]].z );
#endif
                     

        /* 2nd vertex */
        GXPosition1x16(indices[1]);     /* Position */
#if 1
        GXNormal1x16(indices[1]);       /* Normal */
#endif
        if (rxGEOMETRY_PRELIT & flags) GXColor1x16(indices[1]);
        if (rxGEOMETRY_TEXTURED & flags) GXTexCoord1x16(indices[1]);
        if (rpGEOMETRYTEXTURED2 & flags) GXTexCoord1x16(indices[1]);
      
#if 0
        GXNormal3f32(((RwV3d*)resEntryHeader->normals)[indices[1]].x,
                     ((RwV3d*)resEntryHeader->normals)[indices[1]].y,
                     ((RwV3d*)resEntryHeader->normals)[indices[1]].z);
                     
        GXNormal3f32(binormals[indices[1]].x,
                     binormals[indices[1]].y,
                     binormals[indices[1]].z );
                     
        GXNormal3f32(tangents[indices[1]].x,
                     tangents[indices[1]].y,
                     tangents[indices[1]].z );
#endif                     
        

        /* 3rd vertex */
        GXPosition1x16(indices[2]);     /* Position */
#if 1
        GXNormal1x16(indices[2]);       /* Normal */
#endif
        if (rxGEOMETRY_PRELIT & flags) GXColor1x16(indices[2]);
        if (rxGEOMETRY_TEXTURED & flags) GXTexCoord1x16(indices[2]);
        if (rpGEOMETRYTEXTURED2 & flags) GXTexCoord1x16(indices[2]);
      
#if 0
        GXNormal3f32(((RwV3d*)resEntryHeader->normals)[indices[2]].x,
                     ((RwV3d*)resEntryHeader->normals)[indices[2]].y,
                     ((RwV3d*)resEntryHeader->normals)[indices[2]].z);
                     
        GXNormal3f32(binormals[indices[2]].x,
                     binormals[indices[2]].y,
                     binormals[indices[2]].z );
                     
        GXNormal3f32(tangents[indices[2]].x,
                     tangents[indices[2]].y,
                     tangents[indices[2]].z );
#endif
    }
    GXEnd();

    /*
     * Free the allocated arrays
     */
    RwFree(binormals);
    RwFree(tangents);


    RWRETURN(object);
}



static void *
_rpGCMatFxAtomicRender(void *object, RxGameCubePipeData *pipeData,
                       RxGameCubeInstanceData *instancedData,
                       RxGameCubeResEntryHeader *resEntryHeader)
{
    RwV3d               *up, *at, *right, *pos;
    RwInt32             flags;
    RwMatrix            transformMatrix;
    Mtx                 mtx;

    RWFUNCTION(RWSTRING("_rpGCMatFxRenderCallback"));

    /*
     * Calculate the transform matrix.
     */
    if (rpATOMIC == RwObjectGetType(object))
    {
        /* Calculate the transform matrix */
        RwMatrix tmp;

        RwMatrixSetIdentity(&transformMatrix);
        RwMatrixSetIdentity(&tmp);

        RwMatrixInvert(&tmp, RwFrameGetLTM(RwCameraGetFrame(RWSRCGLOBAL(curCamera))));
        RwMatrixMultiply(&transformMatrix,
                         RwFrameGetLTM(RpAtomicGetFrame((RpAtomic *)object)),
                         &tmp);
    }
    else
    {
        RwMatrixInvert(&transformMatrix,
                   RwFrameGetLTM(RwCameraGetFrame(RWSRCGLOBAL(curCamera))));
    }

    /*
	 * Set the transform matrix
	 */
    right = &transformMatrix.right;
    up = &transformMatrix.up;
    at = &transformMatrix.at;
    pos = &transformMatrix.pos;

    mtx[0][0] = -right->x;
    mtx[0][1] = -up->x;
    mtx[0][2] = -at->x;
    mtx[0][3] = -pos->x;

    mtx[1][0] = right->y;
    mtx[1][1] = up->y;
    mtx[1][2] = at->y;
    mtx[1][3] = pos->y;

    mtx[2][0] = -right->z;
    mtx[2][1] = -up->z;
    mtx[2][2] = -at->z;
    mtx[2][3] = -pos->z;

    GXLoadPosMtxImm(mtx, GX_PNMTX0);

    flags = pipeData->flags;

    if (rxGEOMETRY_NORMALS & flags)
	{
        GXLoadNrmMtxImm(mtx, GX_PNMTX0);
	}

    /* Use GX_PNMTX0 as the default matrix ID */
    GXSetCurrentMtx(GX_PNMTX0);

    /*
     * Setup the lights
     */
    //_rxGCHWLightsSetup(object, pipeData);


    /*
     * Vertex setup
     */

    /* Set vertex descriptor to GX_NONE */
    GXClearVtxDesc();

    /* Position */
    GXSetVtxDesc(GX_VA_POS, GX_INDEX16); /* 16-bit indices */
    GXSetVtxAttrFmt(GX_VTXFMT0,
                    GX_VA_POS,
                    GX_POS_XYZ,
                    GX_F32,
                    0);
    RWASSERT(NULL != resEntryHeader->positions);
    GXSetArray(GX_VA_POS, (void *)(resEntryHeader->positions), sizeof(RwV3d));

    /* Normals */
    if ((rxGEOMETRY_NORMALS & flags) && (rxGEOMETRY_LIGHT & flags))
    {
        GXSetVtxDesc(GX_VA_NRM, GX_INDEX16); /* 16-bit indices */
        GXSetVtxAttrFmt(GX_VTXFMT0,
                        GX_VA_NRM,
                        GX_NRM_XYZ,
                        GX_F32,
                        0);
        RWASSERT(NULL != resEntryHeader->normals);
        GXSetArray(GX_VA_NRM, (void *)(resEntryHeader->normals), sizeof(RwV3d));
    }

    /* Pre-lighting */
    if (rxGEOMETRY_PRELIT & flags)
    {
        /* Color */
        GXSetVtxDesc(GX_VA_CLR0, GX_INDEX16); /* 16-bit indices */
        GXSetVtxAttrFmt(GX_VTXFMT0,
                        GX_VA_CLR0,
                        GX_CLR_RGB,
                        GX_RGB8,
                        0);
        RWASSERT(NULL != resEntryHeader->preLight);
        GXSetArray(GX_VA_CLR0, (void *)(resEntryHeader->preLight), sizeof(RwRGBA));
    }

    /* Texture coordinates */
    if (rxGEOMETRY_TEXTURED & flags)        
    {
        /* UVs */
        GXSetVtxDesc(GX_VA_TEX0, GX_INDEX16); /* 16-bit indices */
        GXSetVtxAttrFmt(GX_VTXFMT0,
                        GX_VA_TEX0,
                        GX_TEX_ST,
                        GX_F32,
                        0);
        RWASSERT(NULL != resEntryHeader->texCoords[0]);
        GXSetArray(GX_VA_TEX0, (void *)(resEntryHeader->texCoords[0]), sizeof(RwTexCoords));
    }

    /* Texture coordinates */
    if (rpGEOMETRYTEXTURED2 & flags)        
    {
        /* UVs */
        GXSetVtxDesc(GX_VA_TEX0, GX_INDEX16); /* 16-bit indices */
        GXSetVtxAttrFmt(GX_VTXFMT0,
                        GX_VA_TEX0,
                        GX_TEX_ST,
                        GX_F32,
                        0);
        RWASSERT(NULL != resEntryHeader->texCoords[0]);
        GXSetArray(GX_VA_TEX0, (void *)(resEntryHeader->texCoords[0]), sizeof(RwTexCoords));

        /* UVs */
        GXSetVtxDesc(GX_VA_TEX1, GX_INDEX16); /* 16-bit indices */
        GXSetVtxAttrFmt(GX_VTXFMT0,
                        GX_VA_TEX1,
                        GX_TEX_ST,
                        GX_F32,
                        0);
        RWASSERT(NULL != resEntryHeader->texCoords[1]);
        GXSetArray(GX_VA_TEX1, (void *)(resEntryHeader->texCoords[1]), sizeof(RwTexCoords));
    }

    /*
     * Renderstates
     */

    if (!(rxGEOMETRY_PRELIT & pipeData->flags) &&
         (rxGEOMETRY_MODULATE & pipeData->flags))
    {
        if (pipeData->numLights > 0)
        {
            /*
             * Set material color in GXSetChanCtrl() to RpMaterial color.
             */
            GXSetChanMatColor(GX_COLOR0A0,
                              *(GXColor *)&instancedData->material->color);
        }
        else if (TRUE == pipeData->ambientLight)
        {
            GXColor color;

            /*
             * Modulate ambient color with material color per material and set
             * as material color in GXSetChanCtrl().
             */
            color.r = (RwUInt8)((RwReal)instancedData->material->color.red *
                                pipeData->ambientLightColor.red);
            color.g = (RwUInt8)((RwReal)instancedData->material->color.green *
                                pipeData->ambientLightColor.green);
            color.b = (RwUInt8)((RwReal)instancedData->material->color.blue *
                                pipeData->ambientLightColor.blue);
            color.a = instancedData->material->color.alpha;

            GXSetChanMatColor(GX_COLOR0A0, color);
        }        
    }

    /*
     * Execute the display list
     */
    GXCallDisplayList(instancedData->displayList, instancedData->size);        

    RWRETURN(object);
}

/****************************************************************************
 rpGameCubeAtomicMatFXRenderCallback
 
 Purpose:
 
 On entry:
                
 On exit:
 */
static void*
_rpGameCubeAtomicMatFXRenderCallback(void* object, 
                                     RxGameCubePipeData *data)
{
    RxGameCubeResEntryHeader    *resEntryHeader;
    RxGameCubeInstanceData      *instancedData;
    RwUInt32                    flags;
    RwUInt32                    numMeshes;
    GXColor                     AmbCol = {0x00, 0x00, 0x00, 0xff};
    rpMatFXMaterialData         *matFXData;
    MatFXEnvMapData             *envMapData;
    MatFXDualData               *dualData;
    MatFXBumpMapData            *bumpMapData;

	RWFUNCTION(RWSTRING("_rpGamecubeAtomicMatFXRenderCallback"));

    resEntryHeader = (RxGameCubeResEntryHeader *)(data->resEntry + 1);
    RWASSERT(resEntryHeader);

    instancedData = (RxGameCubeInstanceData *)(resEntryHeader + 1);
    RWASSERT(instancedData);

    flags = data->flags;
    numMeshes = resEntryHeader->numMeshes;

    matFXData = *MATFXMATERIALGETDATA( instancedData->material );

    _rpGCMatFxAtomicLightingCallback( object, data ); 

    _rpGCMatFxTEVHWLightsSetup( data ); 

    while(numMeshes--)
    {
        if(matFXData)
        {
            switch(matFXData->flags)
            {
                case rpMATFXEFFECTENVMAP:                    

                    envMapData = MATFXGCNENVMAPGETDATA( instancedData->material );
                    _rpGCMatFxEnvMapTEVSetup( instancedData->material, data, (RpAtomic*)object );

                    _rpGCMatFxSetTexture( instancedData->material->texture, flags, 0 );
                    _rpGCMatFxSetTexture( envMapData->texture, flags, 1 );
                    _rpGCMatFxAtomicRender( object, data, instancedData, resEntryHeader );
                    break;

                case rpMATFXEFFECTBUMPMAP:

                    bumpMapData = MATFXGCNBUMPMAPGETDATA( instancedData->material );
                    _rpGCMatFxBumpMapTEVSetup( data );
                    _rpGCMatFxSetTexture( instancedData->material->texture, flags, 0 );                     
                    _rpGCMatFxSetTexture( bumpMapData->texture, flags, 1 );
                    _rpGCMatFxAtomicDirectRender( object, 
                                                  data, 
                                                  instancedData, 
                                                  resEntryHeader,  
                                                  bumpMapData );
                    break;

                case rpMATFXEFFECTBUMPENVMAP:
                    
                    bumpMapData = MATFXGCNBUMPMAPGETDATA( instancedData->material );
                    /*_rpGCMatFxSetTexture(bumpMapData->texture, flags);*/

                    envMapData = MATFXGCNBUMPENVMAPGETDATA( instancedData->material );
                    /*_rpGCMatFxSetTexture(envMapData->texture, flags);*/

                    _rpGCMatFxBumpEnvMapTEVSetup( data );

                    _rpGCMatFxSetTexture( instancedData->material->texture, flags, 0 );
                    _rpGCMatFxAtomicDirectRender( object, 
                                                  data, 
                                                  instancedData, 
                                                  resEntryHeader,
                                                  bumpMapData );
                    break;

                case rpMATFXEFFECTDUAL:

                    dualData = MATFXGCNDUALPASSGETDATA( instancedData->material );
                    _rpGCMatFxDualPassTEVSetup( data );
                    _rpGCMatFxSetTexture( instancedData->material->texture, flags, 0 );
                    _rpGCMatFxSetTexture( dualData->texture, flags, 1 );
                    _rpGCMatFxAtomicRender( object, data, instancedData, resEntryHeader );
                    break;
            }
        }        

        instancedData++;
     
    }
                       

	RWRETURN(object);
}

/*--- Create and destory pipelines ------------------------------------------*/

RwBool
_rpMatFXPipelinesCreate(void)
{
    RxPipeline  *pipe;

    RWFUNCTION(RWSTRING("_rpTeamMatFXGCDualPassPipelineCreate"));

    pipe = RxPipelineCreate();

    if (pipe)
    {
        RxLockedPipe    *lpipe;
        RxPipelineNode  *plnode;

        if ((lpipe = RxPipelineLock(pipe)) != NULL)
        {
            lpipe = RxLockedPipeAddFragment(lpipe, NULL,
                                           RxNodeDefinitionGetGameCubeAtomicAllInOne(),
                                           NULL);
            plnode = RxPipelineFindNodeByName( lpipe,
                                               "GameCubeAtomicAllInOne.csl",
                                               (RxPipelineNode *)NULL,
                                               (RwInt32 *)NULL );

            lpipe = RxLockedPipeUnlock(lpipe);

            RWASSERT(pipe == (RxPipeline *)lpipe);

            if (lpipe != NULL)
            {
                RxGameCubeAllInOneSetRenderCallBack(plnode,
                    _rpGameCubeAtomicMatFXRenderCallback);

                MatFXAtomicPipe = pipe;
                
                RWRETURN(TRUE);
            }
        }

        RxPipelineDestroy(pipe);

        pipe = NULL;
    }

    RWRETURN(TRUE);
}

RwBool
_rpMatFXPipelinesDestroy(void)
{
    RWFUNCTION(RWSTRING("_rpMatFXPipelinesDestroy"));

    if (MatFXAtomicPipe)
    {
        RxPipelineDestroy(MatFXAtomicPipe);
        MatFXAtomicPipe = NULL;
    }

    RWRETURN(TRUE);
}

/*--- Attach pipelines ------------------------------------------------------*/
RpAtomic           *
_rpMatFXPipelineAtomicSetup(RpAtomic * atomic)
{
    RWFUNCTION(RWSTRING("_rpMatFXPipelineAtomicSetup"));
    RWASSERT(atomic);

    RpAtomicSetInstancePipeline(atomic, MatFXAtomicPipe);

    RWRETURN(atomic);
}

RpWorldSector      *
_rpMatFXPipelineWorldSectorSetup(RpWorldSector * worldSector)
{
    RWFUNCTION(RWSTRING("_rpMatFXPipelineWorldSectorSetup"));
    RWASSERT(worldSector);
    RWRETURN(worldSector);
}

/*--- Enable effects --------------------------------------------------------*/
RpMaterial         *
_rpMatFXEnvMapEnable(RpMaterial * material)
{
    RWFUNCTION(RWSTRING("_rpMatFXEnvMapEnable"));
    RWRETURN(material);
}

RpMaterial         *
_rpMatFXBumpMapEnable(RpMaterial * material)
{
    RWFUNCTION(RWSTRING("_rpMatFXBumpMapEnable"));
    RWRETURN(material);
}

RpMaterial         *
_rpMatFXDualEnable(RpMaterial * material)
{
    RWFUNCTION(RWSTRING("_rpMatFXDualEnable"));
    RWRETURN(material);
}

/*--- Upload texture --------------------------------------------------------*/

/*--- Device data fucntions -------------------------------------------------*/
RwBool
_rpMatFXSetupDualRenderState(MatFXDualData * __RWUNUSEDRELEASE__ dualData,
                             RwRenderState __RWUNUSED__ nState)
{
    RWFUNCTION(RWSTRING("_rpMatFXSetupDualRenderState"));
    RWASSERT(dualData);
    RWRETURN(TRUE);
}

RwTexture *
_rpMatFXSetupBumpMapTexture(const RwTexture *baseTexture,
                            const RwTexture *effectTexture)
{
    RwTexture *texture;
    RWFUNCTION(RWSTRING("_rpMatFXSetupBumpMapTexture"));

    texture = _rpMatFXTextureMaskCreate(baseTexture, effectTexture);

    RWRETURN(texture);
}

