/*
 * World toolkit.
 */
/***************************************************************************
 *                                                                         *
 * Module  : rtworld.h                                                     *
 *                                                                         *
 * Purpose : World tool helper functions header                            *
 *                                                                         *
 **************************************************************************/

#ifndef RTWORLD_H
#define RTWORLD_H

/**
 * \defgroup rtworld RtWorld
 * \ingroup rttool
 *
 * World Import Toolkit for Renderware.
 */

/****************************************************************************
 Includes
 */

#include "rpworld.h"

#include "rtworld.rpe"    /* automatically generated header file */

/****************************************************************************
 Defines
 */

/* RWPUBLIC */
/****************************************************************************
 Global Types
 */

/**
 * \ingroup rtworld
 * RtWorldForAllPolygonsCallBack 
 * \ref RtWorldForAllPolygonsCallBack 
 * is the callback function supplied to \ref RtWorldForAllPolygons.
 * 
 * Optionally, the callback will be passed a pointer to a user-defined
 * data structure.If no such structure was specified, this will be NULL.
 * 
 * Note that the header file rtworld.h must be included, and the program
 * should be linked to rtworld.lib.
 * 
 * \param  world         Pointer to the world.
 * \param  worldSector   Pointer to the world sector.
 * \param  index         The index of the polygon.
 * \param  worldVertex   Pointer to array of polygon vertices.
 * \param  material      Pointer to the polygon's material.
 * \param  pData         Pointer to user data structure.
 * 
 * \return
 * Returns a pointer to the world if successful or NULL if an error occurred.
 * 
 * \see RtWorldForAllPolygons
 *
 */
typedef RpWorld * (* RtWorldForAllPolygonsCallBack)(RpWorld         *world,
                                                    RpWorldSector   *worldSector,
                                                    RwInt32         index,
                                                    RpWorldVertex   *worldVertex,
                                                    RpMaterial      *material,
                                                    void            *pData);

/* RWPUBLICEND */

/****************************************************************************
 Exported globals
 */

/* RWPUBLIC */
/****************************************************************************
 Function prototypes
 */

#ifdef    __cplusplus
extern "C"
{
#endif                          /* __cplusplus */

/* Get the number of various things in the world */
extern RwInt32 RtWorldGetNumWorldSectors(RpWorld *world);
extern RwInt32 RtWorldGetNumVertices(RpWorld *world);
extern RwInt32 RtWorldGetNumPolygons(RpWorld *world);

/* Enumerate all polygons in the world */
extern RpWorld *RtWorldForAllPolygons(RpWorld *world,
                                        RtWorldForAllPolygonsCallBack fpCallBack,
                                        void *pData);

/* Find things out about materials in the world */
extern RwInt32 RtWorldFindMaterialNum(RpWorld *world, RpMaterial *material);
extern RpMaterial *RtWorldFindMaterialWithTextureName(RpWorld *world, RwChar *name);

/* Optimisation functions in optimise.c */
extern RpClump *RtClumpOptimize(RpClump *clump, RwReal dist);
extern RpAtomic *RtAtomicOptimize(RpAtomic *atomic, RwReal dist);
extern RpGeometry *RtGeometryOptimize(RpGeometry *geometry, RwReal dist);

/* Import utility functions from imputil.c */
extern RpGeometry *RtGeometryCalculateVertexNormals(RpGeometry *geometry);

#ifdef    __cplusplus
}
#endif                          /* __cplusplus */

/* RWPUBLICEND */

#endif /* RWTLWRLD_H */

