/*
 * Data structures for rtbezpat toolkit
 * Copyright (c) Criterion Software Limited
 */

#if (!defined(RTBEZPAT_H))
#define RTBEZPAT_H

/**
 * \defgroup rtbezpatch RtBezier
 * \ingroup rttool
 *
 * Bezier Patches
 */

/**
 * \ingroup rtbezpatch
 * \typedef RtBezierRow
 *  typedef for a difference row.
 * RtBezierRow is an  array of 4 difference vectors
 */
typedef RwV4d       RtBezierRow[4];

/**
 * \ingroup rtbezpatch
 * \typedef RtBezierMatrix
 *  typedef for a difference matrix
 *  RtBezierMatrix ia an array of 4 difference rows
 */
typedef RtBezierRow RtBezierMatrix[4];


#define RtBezierQuadMatrixEvaluateMacro(_out, _B, _u, _v)                  \
MACRO_START                                                                \
{                                                                          \
    (_out)->x =                                                            \
        ((((((((_B)[0][0].x) * (_v) + (_B)[0][1].x) * (_v) +               \
             (_B)[0][2].x) * (_v) + (_B)[0][3].x)) * (_u) +                \
          (((((_B)[1][0].x) * (_v) + (_B)[1][1].x) * (_v) +                \
            (_B)[1][2].x) * (_v) + (_B)[1][3].x)) * (_u) +                 \
         (((((_B)[2][0].x) * (_v) + (_B)[2][1].x) * (_v) +                 \
           (_B)[2][2].x) * (_v) + (_B)[2][3].x)) * (_u) +                  \
        (((((_B)[3][0].x) * (_v) +                                         \
           (_B)[3][1].x) * (_v) +                                          \
          (_B)[3][2].x) * (_v) +                                           \
         (_B)[3][3].x);                                                    \
                                                                           \
    (_out)->y =                                                            \
        ((((((((_B)[0][0].y) * (_v) + (_B)[0][1].y) * (_v) +               \
             (_B)[0][2].y) * (_v) + (_B)[0][3].y)) * (_u) +                \
          (((((_B)[1][0].y) * (_v) + (_B)[1][1].y) * (_v) +                \
            (_B)[1][2].y) * (_v) + (_B)[1][3].y)) * (_u) +                 \
         (((((_B)[2][0].y) * (_v) + (_B)[2][1].y) * (_v) +                 \
           (_B)[2][2].y) * (_v) + (_B)[2][3].y)) * (_u) +                  \
        (((((_B)[3][0].y) * (_v) +                                         \
           (_B)[3][1].y) * (_v) +                                          \
          (_B)[3][2].y) * (_v) +                                           \
         (_B)[3][3].y);                                                    \
                                                                           \
    (_out)->z =                                                            \
        ((((((((_B)[0][0].z) * (_v) + (_B)[0][1].z) * (_v) +               \
             (_B)[0][2].z) * (_v) + (_B)[0][3].z)) * (_u) +                \
          (((((_B)[1][0].z) * (_v) + (_B)[1][1].z) * (_v) +                \
            (_B)[1][2].z) * (_v) + (_B)[1][3].z)) * (_u) +                 \
         (((((_B)[2][0].z) * (_v) + (_B)[2][1].z) * (_v) +                 \
           (_B)[2][2].z) * (_v) + (_B)[2][3].z)) * (_u) +                  \
        (((((_B)[3][0].z) * (_v) +                                         \
           (_B)[3][1].z) * (_v) +                                          \
          (_B)[3][2].z) * (_v) +                                           \
         (_B)[3][3].z);                                                    \
                                                                           \
}                                                                          \
MACRO_STOP

#define RtBezierQuadMatrixEvaluateV4dMacro(_out, _B, _u, _v)               \
MACRO_START                                                                \
{                                                                          \
    RtBezierQuadMatrixEvaluateMacro((_out), (_B), (_u), (_v));             \
                                                                           \
    (_out)->w =                                                            \
        ((((((((_B)[0][0].w) * (_v) + (_B)[0][1].w) * (_v) +               \
             (_B)[0][2].w) * (_v) + (_B)[0][3].w)) * (_u) +                \
          (((((_B)[1][0].w) * (_v) + (_B)[1][1].w) * (_v) +                \
            (_B)[1][2].w) * (_v) + (_B)[1][3].w)) * (_u) +                 \
         (((((_B)[2][0].w) * (_v) + (_B)[2][1].w) * (_v) +                 \
           (_B)[2][2].w) * (_v) + (_B)[2][3].w)) * (_u) +                  \
        (((((_B)[3][0].w) * (_v) +                                         \
           (_B)[3][1].w) * (_v) +                                          \
          (_B)[3][2].w) * (_v) +                                           \
         (_B)[3][3].w);                                                    \
}                                                                          \
MACRO_STOP

#define RtBezierQuadDifferenceStepUMacro(_row)                             \
MACRO_START                                                                \
{                                                                          \
    (_row)[0].x += (_row)[1].x;                                            \
    (_row)[0].y += (_row)[1].y;                                            \
    (_row)[0].z += (_row)[1].z;                                            \
    (_row)[0].w += (_row)[1].w;                                            \
                                                                           \
    (_row)[1].x += (_row)[2].x;                                            \
    (_row)[1].y += (_row)[2].y;                                            \
    (_row)[1].z += (_row)[2].z;                                            \
    (_row)[1].w += (_row)[2].w;                                            \
                                                                           \
    (_row)[2].x += (_row)[3].x;                                            \
    (_row)[2].y += (_row)[3].y;                                            \
    (_row)[2].z += (_row)[3].z;                                            \
    (_row)[2].w += (_row)[3].w;                                            \
                                                                           \
}                                                                          \
MACRO_STOP

#define RtBezierQuadDifferenceStepVMacro(_mat)                             \
MACRO_START                                                                \
{                                                                          \
    /* (_row) 0 */                                                         \
    (_mat)[0][0].x += (_mat)[1][0].x;                                      \
    (_mat)[0][0].y += (_mat)[1][0].y;                                      \
    (_mat)[0][0].z += (_mat)[1][0].z;                                      \
    (_mat)[0][0].w += (_mat)[1][0].w;                                      \
                                                                           \
    (_mat)[0][1].x += (_mat)[1][1].x;                                      \
    (_mat)[0][1].y += (_mat)[1][1].y;                                      \
    (_mat)[0][1].z += (_mat)[1][1].z;                                      \
    (_mat)[0][1].w += (_mat)[1][1].w;                                      \
                                                                           \
    (_mat)[0][2].x += (_mat)[1][2].x;                                      \
    (_mat)[0][2].y += (_mat)[1][2].y;                                      \
    (_mat)[0][2].z += (_mat)[1][2].z;                                      \
    (_mat)[0][2].w += (_mat)[1][2].w;                                      \
                                                                           \
    (_mat)[0][3].x += (_mat)[1][3].x;                                      \
    (_mat)[0][3].y += (_mat)[1][3].y;                                      \
    (_mat)[0][3].z += (_mat)[1][3].z;                                      \
    (_mat)[0][3].w += (_mat)[1][3].w;                                      \
                                                                           \
    /* (_row) 1 */                                                         \
    (_mat)[1][0].x += (_mat)[2][0].x;                                      \
    (_mat)[1][0].y += (_mat)[2][0].y;                                      \
    (_mat)[1][0].z += (_mat)[2][0].z;                                      \
    (_mat)[1][0].w += (_mat)[2][0].w;                                      \
                                                                           \
    (_mat)[1][1].x += (_mat)[2][1].x;                                      \
    (_mat)[1][1].y += (_mat)[2][1].y;                                      \
    (_mat)[1][1].z += (_mat)[2][1].z;                                      \
    (_mat)[1][1].w += (_mat)[2][1].w;                                      \
                                                                           \
    (_mat)[1][2].x += (_mat)[2][2].x;                                      \
    (_mat)[1][2].y += (_mat)[2][2].y;                                      \
    (_mat)[1][2].z += (_mat)[2][2].z;                                      \
    (_mat)[1][2].w += (_mat)[2][2].w;                                      \
                                                                           \
    (_mat)[1][3].x += (_mat)[2][3].x;                                      \
    (_mat)[1][3].y += (_mat)[2][3].y;                                      \
    (_mat)[1][3].z += (_mat)[2][3].z;                                      \
    (_mat)[1][3].w += (_mat)[2][3].w;                                      \
                                                                           \
    /* (_row) 2 */                                                         \
    (_mat)[2][0].x += (_mat)[3][0].x;                                      \
    (_mat)[2][0].y += (_mat)[3][0].y;                                      \
    (_mat)[2][0].z += (_mat)[3][0].z;                                      \
    (_mat)[2][0].w += (_mat)[3][0].w;                                      \
                                                                           \
    (_mat)[2][1].x += (_mat)[3][1].x;                                      \
    (_mat)[2][1].y += (_mat)[3][1].y;                                      \
    (_mat)[2][1].z += (_mat)[3][1].z;                                      \
    (_mat)[2][1].w += (_mat)[3][1].w;                                      \
                                                                           \
    (_mat)[2][2].x += (_mat)[3][2].x;                                      \
    (_mat)[2][2].y += (_mat)[3][2].y;                                      \
    (_mat)[2][2].z += (_mat)[3][2].z;                                      \
    (_mat)[2][2].w += (_mat)[3][2].w;                                      \
                                                                           \
    (_mat)[2][3].x += (_mat)[3][3].x;                                      \
    (_mat)[2][3].y += (_mat)[3][3].y;                                      \
    (_mat)[2][3].z += (_mat)[3][3].z;                                      \
    (_mat)[2][3].w += (_mat)[3][3].w;                                      \
}                                                                          \
MACRO_STOP


#define RtBezierTriMatrixEvaluateMacro(_out, _P, _u, _v)                      \
MACRO_START                                                                   \
{                                                                             \
    const RwReal        u1 = (_u);                                            \
    const RwReal        u2 = u1 * u1;                                         \
    const RwReal        u3 = u2 * u1;                                         \
    const RwReal        v1 = (_v);                                            \
    const RwReal        v2 = v1 * v1;                                         \
    const RwReal        v3 = v2 * v1;                                         \
                                                                              \
    (_out)->x =                                                               \
        (3*(_P)[1][0].x - (_P)[0][0].x - 3*(_P)[2][0].x + (_P)[3][0].x)*u3            \
        + (v1*(3*(_P)[0][1].x - 3*(_P)[0][0].x + 6*(_P)[1][0].x                     \
               - 6*(_P)[1][1].x - 3*(_P)[2][0].x + 3*(_P)[2][1].x) +                \
           3*(_P)[0][0].x - 6*(_P)[1][0].x + 3*(_P)[2][0].x)*u2                     \
        + u1*((6*(_P)[0][1].x - 3*(_P)[0][0].x - 3*(_P)[0][2].x                     \
               + 3*(_P)[1][0].x - 6*(_P)[1][1].x + 3*(_P)[1][2].x)*v2 +             \
              v1*(6*(_P)[0][0].x - 6*(_P)[0][1].x - 6*(_P)[1][0].x + 6*(_P)[1][1].x) -\
              3*(_P)[0][0].x + 3*(_P)[1][0].x)                                    \
        + (3*(_P)[0][1].x - (_P)[0][0].x - 3*(_P)[0][2].x + (_P)[0][3].x)*v3 +        \
        (3*(_P)[0][0].x - 6*(_P)[0][1].x + 3*(_P)[0][2].x)*v2 +                     \
        v1*(3*(_P)[0][1].x - 3*(_P)[0][0].x) + (_P)[0][0].x;                        \
                                                                              \
    (_out)->y =                                                               \
        (3*(_P)[1][0].y - (_P)[0][0].y - 3*(_P)[2][0].y + (_P)[3][0].y)*u3            \
        + (v1*(3*(_P)[0][1].y - 3*(_P)[0][0].y + 6*(_P)[1][0].y                     \
               - 6*(_P)[1][1].y - 3*(_P)[2][0].y + 3*(_P)[2][1].y) +                \
           3*(_P)[0][0].y - 6*(_P)[1][0].y + 3*(_P)[2][0].y)*u2                     \
        + u1*((6*(_P)[0][1].y - 3*(_P)[0][0].y - 3*(_P)[0][2].y                     \
               + 3*(_P)[1][0].y - 6*(_P)[1][1].y + 3*(_P)[1][2].y)*v2 +             \
              v1*(6*(_P)[0][0].y - 6*(_P)[0][1].y - 6*(_P)[1][0].y + 6*(_P)[1][1].y) -\
              3*(_P)[0][0].y + 3*(_P)[1][0].y)                                    \
        + (3*(_P)[0][1].y - (_P)[0][0].y - 3*(_P)[0][2].y + (_P)[0][3].y)*v3 +        \
        (3*(_P)[0][0].y - 6*(_P)[0][1].y + 3*(_P)[0][2].y)*v2 +                     \
        v1*(3*(_P)[0][1].y - 3*(_P)[0][0].y) + (_P)[0][0].y;                        \
                                                                              \
    (_out)->z =                                                               \
        (3*(_P)[1][0].z - (_P)[0][0].z - 3*(_P)[2][0].z + (_P)[3][0].z)*u3            \
        + (v1*(3*(_P)[0][1].z - 3*(_P)[0][0].z + 6*(_P)[1][0].z                     \
               - 6*(_P)[1][1].z - 3*(_P)[2][0].z + 3*(_P)[2][1].z) +                \
           3*(_P)[0][0].z - 6*(_P)[1][0].z + 3*(_P)[2][0].z)*u2                     \
        + u1*((6*(_P)[0][1].z - 3*(_P)[0][0].z - 3*(_P)[0][2].z                     \
               + 3*(_P)[1][0].z - 6*(_P)[1][1].z + 3*(_P)[1][2].z)*v2 +             \
              v1*(6*(_P)[0][0].z - 6*(_P)[0][1].z - 6*(_P)[1][0].z + 6*(_P)[1][1].z) -\
              3*(_P)[0][0].z + 3*(_P)[1][0].z)                                    \
        + (3*(_P)[0][1].z - (_P)[0][0].z - 3*(_P)[0][2].z + (_P)[0][3].z)*v3 +        \
        (3*(_P)[0][0].z - 6*(_P)[0][1].z + 3*(_P)[0][2].z)*v2 +                     \
        v1*(3*(_P)[0][1].z - 3*(_P)[0][0].z) + (_P)[0][0].z;                        \
}                                                                             \
MACRO_STOP

#define RtBezierTriMatrixEvaluateV4dMacro(_out, _P, _u, _v)                   \
MACRO_START                                                                   \
{                                                                             \
    const RwReal        u1 = (_u);                                            \
    const RwReal        u2 = u1 * u1;                                         \
    const RwReal        u3 = u2 * u1;                                         \
    const RwReal        v1 = (_v);                                            \
    const RwReal        v2 = v1 * v1;                                         \
    const RwReal        v3 = v2 * v1;                                         \
                                                                              \
    RtBezierTriMatrixEvaluateMacro((_out), (_P), (_u), (_v));                         \
                                                                              \
    (_out)->w =                                                               \
        (3*(_P)[1][0].w - (_P)[0][0].w - 3*(_P)[2][0].w + (_P)[3][0].w)*u3            \
        + (v1*(3*(_P)[0][1].w - 3*(_P)[0][0].w + 6*(_P)[1][0].w                     \
               - 6*(_P)[1][1].w - 3*(_P)[2][0].w + 3*(_P)[2][1].w) +                \
           3*(_P)[0][0].w - 6*(_P)[1][0].w + 3*(_P)[2][0].w)*u2                     \
        + u1*((6*(_P)[0][1].w - 3*(_P)[0][0].w - 3*(_P)[0][2].w                     \
               + 3*(_P)[1][0].w - 6*(_P)[1][1].w + 3*(_P)[1][2].w)*v2 +             \
              v1*(6*(_P)[0][0].w - 6*(_P)[0][1].w - 6*(_P)[1][0].w + 6*(_P)[1][1].w) -\
              3*(_P)[0][0].w + 3*(_P)[1][0].w)                                    \
        + (3*(_P)[0][1].w - (_P)[0][0].w - 3*(_P)[0][2].w + (_P)[0][3].w)*v3 +        \
        (3*(_P)[0][0].w - 6*(_P)[0][1].w + 3*(_P)[0][2].w)*v2 +                     \
        v1*(3*(_P)[0][1].w - 3*(_P)[0][0].w) + (_P)[0][0].w;                        \
}                                                                             \
MACRO_STOP

#define RtBezierTriDifferenceStepUMacro(_row)                              \
MACRO_START                                                                \
{                                                                          \
    (_row)[0].x += (_row)[1].x;                                            \
    (_row)[0].y += (_row)[1].y;                                            \
    (_row)[0].z += (_row)[1].z;                                            \
    (_row)[0].w += (_row)[1].w;                                            \
                                                                           \
    (_row)[1].x += (_row)[2].x;                                            \
    (_row)[1].y += (_row)[2].y;                                            \
    (_row)[1].z += (_row)[2].z;                                            \
    (_row)[1].w += (_row)[2].w;                                            \
                                                                           \
    (_row)[2].x += (_row)[3].x;                                            \
    (_row)[2].y += (_row)[3].y;                                            \
    (_row)[2].z += (_row)[3].z;                                            \
    (_row)[2].w += (_row)[3].w;                                            \
}                                                                          \
MACRO_STOP

#define RtBezierTriDifferenceStepVMacro(_mat)                              \
MACRO_START                                                                \
{                                                                          \
    /* (_row) 0 */                                                         \
    (_mat)[0][0].x += (_mat)[1][0].x;                                      \
    (_mat)[0][0].y += (_mat)[1][0].y;                                      \
    (_mat)[0][0].z += (_mat)[1][0].z;                                      \
    (_mat)[0][0].w += (_mat)[1][0].w;                                      \
                                                                           \
    (_mat)[0][1].x += (_mat)[1][1].x;                                      \
    (_mat)[0][1].y += (_mat)[1][1].y;                                      \
    (_mat)[0][1].z += (_mat)[1][1].z;                                      \
    (_mat)[0][1].w += (_mat)[1][1].w;                                      \
                                                                           \
    (_mat)[0][2].x += (_mat)[1][2].x;                                      \
    (_mat)[0][2].y += (_mat)[1][2].y;                                      \
    (_mat)[0][2].z += (_mat)[1][2].z;                                      \
    (_mat)[0][2].w += (_mat)[1][2].w;                                      \
                                                                           \
                                                                           \
    (_mat)[0][3].x += (_mat)[1][3].x;                                      \
    (_mat)[0][3].y += (_mat)[1][3].y;                                      \
    (_mat)[0][3].z += (_mat)[1][3].z;                                      \
    (_mat)[0][3].w += (_mat)[1][3].w;                                      \
                                                                           \
    /* (_row) 1 */                                                         \
    (_mat)[1][0].x += (_mat)[2][0].x;                                      \
    (_mat)[1][0].y += (_mat)[2][0].y;                                      \
    (_mat)[1][0].z += (_mat)[2][0].z;                                      \
    (_mat)[1][0].w += (_mat)[2][0].w;                                      \
                                                                           \
    (_mat)[1][1].x += (_mat)[2][1].x;                                      \
    (_mat)[1][1].y += (_mat)[2][1].y;                                      \
    (_mat)[1][1].z += (_mat)[2][1].z;                                      \
    (_mat)[1][1].w += (_mat)[2][1].w;                                      \
                                                                           \
    (_mat)[1][2].x += (_mat)[2][2].x;                                      \
    (_mat)[1][2].y += (_mat)[2][2].y;                                      \
    (_mat)[1][2].z += (_mat)[2][2].z;                                      \
    (_mat)[1][2].w += (_mat)[2][2].w;                                      \
                                                                           \
    (_mat)[1][3].x += (_mat)[2][3].x;                                      \
    (_mat)[1][3].y += (_mat)[2][3].y;                                      \
    (_mat)[1][3].z += (_mat)[2][3].z;                                      \
    (_mat)[1][3].w += (_mat)[2][3].w;                                      \
                                                                           \
    /* (_row) 2 */                                                         \
    (_mat)[2][0].x += (_mat)[3][0].x;                                      \
    (_mat)[2][0].y += (_mat)[3][0].y;                                      \
    (_mat)[2][0].z += (_mat)[3][0].z;                                      \
    (_mat)[2][0].w += (_mat)[3][0].w;                                      \
                                                                           \
    (_mat)[2][1].x += (_mat)[3][1].x;                                      \
    (_mat)[2][1].y += (_mat)[3][1].y;                                      \
    (_mat)[2][1].z += (_mat)[3][1].z;                                      \
    (_mat)[2][1].w += (_mat)[3][1].w;                                      \
                                                                           \
    (_mat)[2][2].x += (_mat)[3][2].x;                                      \
    (_mat)[2][2].y += (_mat)[3][2].y;                                      \
    (_mat)[2][2].z += (_mat)[3][2].z;                                      \
    (_mat)[2][2].w += (_mat)[3][2].w;                                      \
                                                                           \
    (_mat)[2][3].x += (_mat)[3][3].x;                                      \
    (_mat)[2][3].y += (_mat)[3][3].y;                                      \
    (_mat)[2][3].z += (_mat)[3][3].z;                                      \
    (_mat)[2][3].w += (_mat)[3][3].w;                                      \
}                                                                          \
MACRO_STOP


#ifdef    __cplusplus
extern "C"
{
#endif         /* __cplusplus */

extern void
RtBezierPatchOpen( void );

extern void
RtBezierQuadMatrixInitialize(RtBezierMatrix B,
                             RtBezierMatrix P);

extern void
RtBezierQuadMatrixWeightSetup(RtBezierMatrix W, 
                              RtBezierMatrix P);

extern void
RtBezierQuadMatrixDifferenceSetup(RtBezierMatrix D, 
                                  RtBezierMatrix W,
                                  RwReal U, 
                                  RwReal V);

extern void
RtBezierQuadMatrixDifferenceInitialize(RtBezierMatrix D,
                                       RtBezierMatrix B,
                                       RwReal U,
                                       RwReal V);

#if ( defined(RWDEBUG) || defined(RWSUPPRESSINLINE) )

extern void
RtBezierQuadMatrixEvaluate(RwV3d * out, RtBezierMatrix B,
                           RwReal u, RwReal v);

extern void
RtBezierQuadMatrixEvaluateV4d(RwV4d * out, RtBezierMatrix B,
                              RwReal u, RwReal v);

extern void
RtBezierQuadDifferenceStepU(RtBezierRow row);

extern void
RtBezierQuadDifferenceStepV(RtBezierMatrix mat);

#else /* ( defined(RWDEBUG) || defined(RWSUPPRESSINLINE) ) */

#define RtBezierQuadMatrixEvaluate(_out, _B, _u, _v) \
        RtBezierQuadMatrixEvaluateMacro(_out, _B, _u, _v)

#define RtBezierQuadMatrixEvaluateV4d(_out, _B, _u, _v) \
        RtBezierQuadMatrixEvaluateV4dMacro(_out, _B, _u, _v)

#define RtBezierQuadDifferenceStepU(_row) \
        RtBezierQuadDifferenceStepUMacro(_row)

#define RtBezierQuadDifferenceStepV(_mat) \
        RtBezierQuadDifferenceStepVMacro(_mat)

#endif /* ( defined(RWDEBUG) || defined(RWSUPPRESSINLINE) ) */

/*
 *
 */

extern void
RtBezierTriMatrixDifferenceInitialize(RtBezierMatrix D,
                                      RtBezierMatrix B,
                                      RwReal U,
                                      RwReal V);


extern void
RtBezierTriMatrixEvaluate(RwV3d * out,
                          RtBezierMatrix B,
                          RwReal u,
                          RwReal v);

extern void
RtBezierTriMatrixEvaluateV4d(RwV4d * out,
                             RtBezierMatrix B,
                             RwReal u,
                             RwReal v);

extern void
RtBezierTriDifferenceStepU(RtBezierRow row);

extern void
RtBezierTriDifferenceStepV(RtBezierMatrix mat);

/* RWPUBLIC */
#ifdef    __cplusplus
}
#endif         /* __cplusplus */

#endif /* (!defined(RTBEZPAT_H)) */

